// $RCSfile$     $Date$     $Revision$
// Copyright (c) 2013 Krodo
// Part of Bylins http://www.mud.ru

#include "help.h"

#include <third_party_libs/fmt/include/fmt/format.h>

#include "obj_prototypes.h"
#include "engine/ui/modify.h"
#include "gameplay/clans/house.h"
#include "gameplay/mechanics/sets_drop.h"
#include "engine/ui/color.h"
#include "global_objects.h"
#include "gameplay/magic/magic_utils.h"
#include "gameplay/mechanics/obj_sets_stuff.h"
#include "gameplay/mechanics/player_races.h"

extern char *help;
extern int top_imrecipes;

namespace PrintActivators {

//  
struct dup_node {
	//  
	std::string clss;
	//  
	std::string afct;
};

void sum_skills(CObjectPrototype::skills_t &target, const CObjectPrototype::skills_t &add) {
	for (auto i : add) {
		if (i.second != 0) {
			auto ii = target.find(i.first);
			if (ii != target.end()) {
				ii->second += i.second;
			} else {
				target[i.first] = i.second;
			}
		}
	}
}

void sum_skills(CObjectPrototype::skills_t &target, const CObjectPrototype::skills_t::value_type &add) {
	if (MUD::Skills().IsValid(add.first) && add.second != 0) {
		auto i = target.find(add.first);
		if (i != target.end()) {
			i->second += add.second;
		} else {
			target[add.first] = add.second;
		}
	}
}

void sum_skills(CObjectPrototype::skills_t &target, const CObjectPrototype *obj) {
	if (obj->has_skills()) {
		CObjectPrototype::skills_t tmp_skills;
		obj->get_skills(tmp_skills);
		sum_skills(target, tmp_skills);
	}
}

inline bool bit_is_set(const uint32_t flags, const int bit) {
	return 0 != (flags & (1 << bit));
}

//   flag_data_by_num()
bool check_num_in_unique_bit_flag_data(const unique_bit_flag_data &data, const int num) {
	return (0 <= num && num < 120) ? data.get_flag(num / 30, 1 << num) : false;
}

std::string print_skill(const CObjectPrototype::skills_t::value_type &skill, bool activ) {
	if (skill.second != 0) {
		char buf[128];

		sprintf(buf, "%s%s%s%s%d%s\r\n",
				(activ ? " +    " : "   "),
				kColorCyn,
				MUD::Skill(skill.first).GetName(),
				(skill.second < 0 ? "   " : "   "),
				abs(skill.second),
				kColorNrm);
		return buf;
	}
	return "";
}

///     " + "  
/// \param header = true (    ' ')
std::string print_skills(const CObjectPrototype::skills_t &skills, bool activ, bool header) {
	std::string out;
	for (const auto &i : skills) {
		out += print_skill(i, activ);
	}

	if (!out.empty() && header) {
		std::string head = activ ? " + " : "   ";
		return head + "  :\r\n" + out;
	}

	return out;
}

//        
std::string print_obj_affects(const CObjectPrototype *const obj) {
	std::stringstream out;

	out << GET_OBJ_PNAME(obj, 0) << "\r\n";

	if (obj->get_no_flags().sprintbits(no_bits, buf2, ",")) {
		out << " : " << buf2 << "\r\n";
	}

	if (GET_OBJ_TYPE(obj) == EObjType::kWeapon) {
		const int drndice = GET_OBJ_VAL(obj, 1);
		const int drsdice = GET_OBJ_VAL(obj, 2);
		out << fmt::format("  '{}D{}'  {:.1}\r\n",
			drndice, drsdice, (drsdice + 1) * drndice / 2.0);
	}

	if (GET_OBJ_TYPE(obj) == EObjType::kWeapon
		|| CAN_WEAR(obj, EWearFlag::kShield)
		|| CAN_WEAR(obj, EWearFlag::kHands)) {
		out << " : " << GET_OBJ_WEIGHT(obj) << "\r\n";
	}

	if (GET_OBJ_AFFECTS(obj).sprintbits(weapon_affects, buf2, ",")) {
		out << " : " << buf2 << "\r\n";
	}

	std::string tmp_str;
	for (int i = 0; i < kMaxObjAffect; i++) {
		if (obj->get_affected(i).modifier != 0) {
			tmp_str += "   " + print_obj_affects(obj->get_affected(i));
		}
	}

	if (!tmp_str.empty()) {
		out << " :\r\n" << tmp_str;
	}

	if (obj->has_skills()) {
		CObjectPrototype::skills_t skills;
		obj->get_skills(skills);
		out << print_skills(skills, false);
	}

	return out.str();
}

//    
std::string print_activator(class_to_act_map::const_iterator &activ, const CObjectPrototype *const obj) {
	std::stringstream out;

	out << " +  :";
	for (const auto &char_class : MUD::Classes()) {
		if (check_num_in_unique_bit_flag_data(activ->first, to_underlying(char_class.GetId()))) {
			if (char_class.IsAvailable()) {
				out << " " << char_class.GetName();
			} else if (char_class.GetId() > ECharClass::kLast && char_class.GetId() <= ECharClass::kNpcLast) {
				out << " ";
			}
		}
	}
/*
 *         - ABYRVALG.
 	for (int i = 0; i <= kNumPlayerClasses * kNumKins; ++i) {
		if (check_num_in_unique_bit_flag_data(activ->first, i)) {
			if (i < kNumPlayerClasses * kNumKins) {
				out << " " << class_name[i];
			} else {
				out << " ";
			}
		}
	}*/
	out << "\r\n";

	FlagData affects = activ->second.get_affects();
	if (affects.sprintbits(weapon_affects, buf2, ",")) {
		out << " +  : " << buf2 << "\r\n";
	}

	std::array<obj_affected_type, kMaxObjAffect> affected = activ->second.get_affected();
	std::string tmp_str;
	for (int i = 0; i < kMaxObjAffect; i++) {
		if (affected[i].modifier != 0) {
			tmp_str += " +    " + print_obj_affects(affected[i]);
		}
	}
	if (!tmp_str.empty()) {
		out << " +  :\r\n" << tmp_str;
	}

	if (GET_OBJ_TYPE(obj) == EObjType::kWeapon) {
		int drndice = 0, drsdice = 0;
		activ->second.get_dices(drsdice, drndice);
		if (drsdice > 0 && drndice > 0) {
			out << fmt::format(" +    '{}D{}'  {:.1}\r\n",
				drndice, drsdice, (drsdice + 1) * drndice / 2.0);
		}
	}

	const int weight = activ->second.get_weight();
	if (weight > 0) {
		out << " +  : " << weight << "\r\n";
	}

	if (activ->second.has_skills()) {
		CObjectPrototype::skills_t skills;
		activ->second.get_skills(skills);
		out << print_skills(skills, true);
	}

	return out.str();
}

////////////////////////////////////////////////////////////////////////////////
struct activators_obj {
	activators_obj() {
		native_no_flag = clear_flags;
		native_affects = clear_flags;
	};

	//      
	std::map<int, clss_activ_node> clss_list;
	//   
	FlagData native_no_flag;
	FlagData native_affects;
	std::vector<obj_affected_type> native_affected;
	CObjectPrototype::skills_t native_skills;

	//   clss_list  
	void fill_class(set_info::const_iterator k);
	//          clss_list
	void fill_node(const set_info &set);
	//  clss_list        
	std::string print();
};

void activators_obj::fill_class(set_info::const_iterator k) {
	for (const auto & m : k->second) {
		for (const auto & q : m.second) {
			for (int i = 0; i <= kNumPlayerClasses * kNumKins; ++i) {
				if (check_num_in_unique_bit_flag_data(q.first, i)) {
					struct clss_activ_node tmp_node;
					clss_list[i] = tmp_node;
				}
			}
		}
	}
}

void activators_obj::fill_node(const set_info &set) {
	for (const auto & k : set) {
		//    
		for (auto & w : clss_list) {
			//   -     
			for (auto m = k.second.rbegin(), mend = k.second.rend(); m != mend; ++m) {
				bool found = false;
				//     
				for (const auto & q : m->second) {
					if (check_num_in_unique_bit_flag_data(q.first, w.first)) {
						//     
						w.second.total_affects += q.second.get_affects();
						sum_apply(w.second.affected, q.second.get_affected());
						// 
						CObjectPrototype::skills_t tmp_skills;
						q.second.get_skills(tmp_skills);
						sum_skills(w.second.skills, tmp_skills);
						found = true;
						break;
					}
				}
				if (found) {
					break;
				}
			}
		}
	}
}

std::string activators_obj::print() {
	std::vector<dup_node> dup_list;

	for (auto & cls_it : clss_list) {
		//    
		dup_node node;
		// ABYRVALG    (  ).          emum
		//node.clss += cls_it.first < kNumPlayerClasses * kNumKins ? class_name[cls_it.first] : "";
		auto class_id = static_cast<ECharClass>(cls_it.first);
		if (MUD::Classes().IsAvailable(class_id)) {
			node.clss += MUD::Class(class_id).GetName();
		} else if (class_id > ECharClass::kLast && class_id <= ECharClass::kNpcLast) {
			node.clss += "";
		}

		// affects
		cls_it.second.total_affects += native_affects;
		if (cls_it.second.total_affects.sprintbits(weapon_affects, buf2, ",")) {
			node.afct += " +  : " + std::string(buf2) + "\r\n";
		}
		// affected
		sum_apply(cls_it.second.affected, native_affected);
		//        
		std::sort(cls_it.second.affected.begin(), cls_it.second.affected.end(),
				  [](const obj_affected_type &lrs, const obj_affected_type &rhs) {
					  return lrs.location < rhs.location;
				  });

		std::string tmp_str;
		for (auto i : cls_it.second.affected) {
			tmp_str += " +    " + print_obj_affects(i);
		}
		if (!tmp_str.empty()) {
			node.afct += " +  :\r\n" + tmp_str;
		}
		// 
		sum_skills(cls_it.second.skills, native_skills);
		node.afct += print_skills(cls_it.second.skills, true);

		//     
		auto i = std::find_if(dup_list.begin(), dup_list.end(),
														 [&](const dup_node &x) {
															 return x.afct == node.afct;
														 });

		if (i != dup_list.end()) {
			i->clss += ", " + node.clss;
		} else {
			dup_list.push_back(node);
		}
	}

	std::string out_str;
	for (const auto & i : dup_list) {
		out_str += " : " + i.clss + "\r\n" + i.afct;
	}
	return out_str;
}
// activators_obj
////////////////////////////////////////////////////////////////////////////////

std::string print_fullset_stats(const set_info &set) {
	std::stringstream out;
	activators_obj activ;

	//   -    +    clss_list
	for (auto k = set.begin(), kend = set.end(); k != kend; ++k) {
		const int rnum = GetObjRnum(k->first);
		if (rnum < 0) {
			continue;
		}
		const auto &obj = obj_proto[rnum];

		//     
		activ.native_no_flag += GET_OBJ_NO(obj);
		activ.native_affects += GET_OBJ_AFFECTS(obj);
		sum_apply(activ.native_affected, obj->get_all_affected());
		sum_skills(activ.native_skills, obj.get());

		//  
		activ.fill_class(k);
	}

	//    
	activ.fill_node(set);

	//  ,  
	out << "  : \r\n";

	if (activ.native_no_flag.sprintbits(no_bits, buf2, ",")) {
		out << " : " << buf2 << "\r\n";
	}

	out << activ.print();

	return out.str();
}

//     
void process() {
	for (const auto & it : ObjData::set_table) {
		std::stringstream out;
		// it->first = int_id, it->second = set_info
		out << "---------------------------------------------------------------------------\r\n";
		out << it.second.get_name() << "\r\n";
		out << "---------------------------------------------------------------------------\r\n";
		out << print_fullset_stats(it.second);
		for (const auto & k : it.second) {
			out << "---------------------------------------------------------------------------\r\n";
			// k->first = int_obj_vnum, k->second = qty_to_camap_map
			const int rnum = GetObjRnum(k.first);
			if (rnum < 0) {
				log("SYSERROR: wrong obj vnum: %d (%s %s %d)", k.first, __FILE__, __func__, __LINE__);
				continue;
			}

			const auto &obj = obj_proto[rnum];
			out << print_obj_affects(obj.get());

			for (const auto & m : k.second) {
				// m->first = num_activators, m->second = class_to_act_map
				for (auto q = m.second.begin(); q != m.second.end(); ++q) {
					out << "  : " << m.first << "\r\n";
					out << print_activator(q, obj.get());
				}
			}
		}
		//    
		std::string set_name = "";
		if (it.second.get_alias().empty()) {
			set_name += it.second.get_name();
			HelpSystem::add_static(utils::EraseAllAny(set_name, " ,."), out.str(), 0, true);
		} else {
			std::string alias = it.second.get_alias();
			for (auto & k : utils::Split(alias, ',')) {
				HelpSystem::add_static(set_name + "" + utils::EraseAllAny(k, " ,."), out.str(), 0, true);
			}
		}
	}
}

} // namespace PrintActivators
using namespace PrintActivators;

////////////////////////////////////////////////////////////////////////////////

namespace HelpSystem {

struct help_node {
	help_node(const std::string &key, const std::string &val)
		: keyword(key), entry(val), min_level(0),
		  sets_drop_page(false), no_immlog(false) {
		utils::ConvertToLow(keyword);
	};

	//   
	std::string keyword;
	//  
	std::string entry;
	//     (   kLevelImmortal)
	int min_level;
	//     
	//      ,    
	bool sets_drop_page;
	//     
	bool no_immlog;
};

// ,      (STATIC)
std::vector<help_node> static_help;
//   ,    
// ,  ,   (DYNAMIC)
std::vector<help_node> dynamic_help;
//      dynamic_help     
bool need_update = false;

const char *HELP_USE_EXMAPLES =
	"&c:&n\r\n"
	"\t\" 3.\"\r\n"
	"\t\" 4.\"\r\n"
	"\t\" \"\r\n"
	"\t\" !\"\r\n"
	"\t\" 3.!\"\r\n"
	"\r\n. : &C&n\r\n";

class UserSearch {
 public:
	explicit UserSearch(CharData *in_ch)
		: strong(false), stop(false), diff_keys(false), level(0), topic_num(0), curr_topic_num(0) { ch = in_ch; };

	//  
	CharData *ch;
	//   (!  )
	bool strong;
	//      
	bool stop;
	//         key_list
	//   1     -    
	//      -    
	bool diff_keys;
	//      
	int level;
	//   ._
	int topic_num;
	//    topic_num != 0
	int curr_topic_num;
	//  
	std::string arg_str;
	//    
	std::vector<std::vector<help_node>::const_iterator> key_list;

	//    search   
	void process(int flag);
	//       
	void search(const std::vector<help_node> &cont);
	//      
	void print_not_found() const;
	//     
	void print_curr_topic(const help_node &node) const;
	//      
	//     key_list  diff_keys
	void print_key_list() const;
};

/// \param min_level = 0, \param no_immlog = false
void add_static(const std::string &key, const std::string &entry,
				int min_level, bool no_immlog) {
	if (key.empty() || entry.empty()) {
		log("SYSERROR: empty str '%s' -> '%s' (%s:%d %s)",
			key.c_str(), entry.c_str(), __FILE__, __LINE__, __func__);
		return;
	}
	std::string tmpentry = utils::SubstStrToUpper(key) + "\r\n\r\n" + entry;
	help_node tmp_node(key, tmpentry);
	tmp_node.min_level = min_level;
	tmp_node.no_immlog = no_immlog;
	static_help.push_back(tmp_node);
}

/// \param min_level = 0, no_immlog = true
void add_dynamic(const std::string &key, const std::string &entry) {
	if (key.empty() || entry.empty()) {
		log("SYSERROR: empty str '%s' -> '%s' (%s:%d %s)",
			key.c_str(), entry.c_str(), __FILE__, __LINE__, __func__);
		return;
	}

	help_node tmp_node(key, entry);
	tmp_node.no_immlog = true;
	dynamic_help.push_back(tmp_node);
}

void add_sets_drop(const std::string &key, const std::string &entry) {
	if (key.empty() || entry.empty()) {
		log("SYSERROR: empty str '%s' -> '%s' (%s %s %d)",
			key.c_str(), entry.c_str(), __FILE__, __func__, __LINE__);
		return;
	}

	help_node tmp_node(key, entry);
	tmp_node.sets_drop_page = true;
	tmp_node.no_immlog = true;

	dynamic_help.push_back(tmp_node);
}
void init_zone_all() {
	std::stringstream out;

	for (std::size_t rnum = 0, i = 1; rnum < zone_table.size(); ++rnum) {
		if (!zone_table[rnum].location.empty()) {
			out << fmt::format("  {:<2} - {}. : {}. : {}.  : {}.\r\n",
					i, zone_table[rnum].name, zone_table[rnum].location,
					zone_table[rnum].group, zone_table[rnum].level);
			++i;
		}
	}
	out << "\r\n. : &C&n";
	add_static("", out.str(), 0, true);
}

std::string OutRecipiesHelp(ECharClass ch_class) {
	std::stringstream out, out2;
	std::string tmpstr;
	int columns = 0, columns2 = 0;
	std::vector<std::string> skills_list;
	std::vector<std::string> skills_list2;

	out << "  :\r\n";
	out2 << "\r\n&G,      :&n\r\n";
	for (int sortpos = 0; sortpos <= top_imrecipes; sortpos++) {
		if (!imrecipes[sortpos].classknow[to_underlying(ch_class)]) {
				continue;
		}
		if (imrecipes[sortpos].remort > 0) {
			skills_list2.push_back(imrecipes[sortpos].name);
			continue;
		}
		skills_list.push_back(imrecipes[sortpos].name);
	}
	utils::SortKoiString(skills_list);
	for (auto it : skills_list) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out << "\t" << std::left << std::setw(30) << it << tmpstr;
	}
	utils::SortKoiString(skills_list2);
	for (auto it : skills_list2) {
			tmpstr = !(++columns2 % 2) ? "\r\n" : "\t";
			out2 << "\t" << "&C" << std::left << std::setw(30) << it << "&n" << tmpstr;
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	if (out2.str().back() == '\t')
		out2 << "\r\n";
	out << out2.str();
	return out.str();
}

std::string OutAllSkillsHelp() {
	std::stringstream out;
	std::vector<std::string> skills_list;
	std::string tmpstr;
	int columns = 0;

	out << "  :\r\n";
	for (const auto &skill : MUD::Skills()) {
		if (skill.IsInvalid())
			continue;
		skills_list.push_back(skill.GetName());
	}
	utils::SortKoiString(skills_list);
	for (auto it : skills_list) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out << fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	return out.str();
}

std::string OutAllRecipiesHelp() {
	std::stringstream out;
	std::vector<std::string> recipies_list;
	std::string tmpstr;
	int columns = 0;

	out << "  :\r\n";
	for (int sortpos = 0; sortpos <= top_imrecipes; sortpos++) {
		recipies_list.push_back(imrecipes[sortpos].name);
	}
	utils::SortKoiString(recipies_list);
	for (auto it : recipies_list) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out << fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	return out.str();
}

std::string OutAllFeaturesHelp() {
	std::stringstream out;
	std::vector<std::string> feats_list;
	std::string tmpstr;
	int columns = 0;

	out << "  :\r\n";
	for (const auto &feat : MUD::Feats()) {
		if (feat.IsUnavailable()) {
			continue;
		}
		feats_list.push_back(MUD::Feat(feat.GetId()).GetCName());
	}
	utils::SortKoiString(feats_list);
	for (auto it : feats_list) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out << fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	return out.str();
}

std::string OutAllSpellsHelp() {
	std::stringstream out;
	std::vector<std::string> spells_list;
	std::string tmpstr;
	int columns = 0;

	out << "  :\r\n";
	for (auto &it : MUD::Spells()) {
		auto spell_id = it.GetId();

		if (MUD::Spell(spell_id).IsInvalid())
			continue;
		spells_list.push_back(it.GetName());
	}
	utils::SortKoiString(spells_list);
	for (auto it : spells_list) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out << fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	return out.str();
}

std::string OutSkillsHelp(ECharClass ch_class) {
	std::stringstream out, out2;
	std::string tmpstr;
	int columns = 0, columns2 = 0;
	std::vector<std::string> skills_list;
	std::vector<std::string> skills_list2;

	out2 << "\r\n&G :&n\r\n";
	for (const auto &skill : MUD::Skills()) {
		if (MUD::Class(ch_class).skills[skill.GetId()].IsInvalid()) {
			continue;
		}
		int num = 0;
		for (const auto &char_class: MUD::Classes()) {
			if (char_class.IsAvailable()) {
				if (char_class.skills[skill.GetId()].IsValid()) {
					++num;
				}
			}
		}
		if (num == 1) {
			skills_list2.push_back(utils::SubstKtoW(skill.GetName()));
			continue;
		}
		skills_list.push_back(utils::SubstKtoW(skill.GetName()));
	}
	std::sort(skills_list.begin(), skills_list.end());
	for (auto it : skills_list) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out << "\t" << std::left << std::setw(30) << utils::SubstWtoK(it) << tmpstr;
	}
	std::sort(skills_list2.begin(), skills_list2.end());
	for (auto it : skills_list2) {
			tmpstr = !(++columns2 % 2) ? "\r\n" : "\t";
			out2 << "\t" << "&C" << std::left << std::setw(30) << utils::SubstWtoK(it) << "&n" << tmpstr;
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	if (out2.str().back() == '\t')
		out2 << "\r\n";
	out << out2.str();
	return out.str();
}

std::string OutMagusSpellsHelp() {
	std::string out, out2, out3;
	std::string tmpstr;
	std::vector<std::string> spells_list;
	std::vector<std::string> spells_list2;
	std::vector<std::string> spells_list3;

	out = " :\r\n";
	out2 = "\r\n&G  -   :&n\r\n";
	out3 = "\r\n&B :&n\r\n";
	for (const auto &spl_info : MUD::Spells()) {
		auto spell_id = spl_info.GetId();
		int num = 0;

		if (MUD::Spell(spell_id).IsInvalid())
			continue;
		if (!spell_create.contains(spell_id))
			continue;
		for (const auto &char_class: MUD::Classes()) {
			if (char_class.IsAvailable()) {
				if (char_class.GetId() == ECharClass::kMagus) 
					continue;
				for (const auto &spell : char_class.spells) {
					if (spell.GetId() == spell_id) {
						++num;
					}
				}
			}
		}
		if (num > 1) {
			spells_list.push_back(MUD::Spell(spell_id).GetCName());
		}
		if (num == 1) {
			spells_list2.push_back(MUD::Spell(spell_id).GetCName());
		}
		if (num == 0) {
			spells_list3.push_back(MUD::Spell(spell_id).GetCName());
		}
	}
	int columns = 0;

	utils::SortKoiString(spells_list);
	for (auto it : spells_list) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	columns = 0;
	utils::SortKoiString(spells_list2);
	for (auto it : spells_list2) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out2 += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	columns = 0;
	utils::SortKoiString(spells_list3);
	for (auto it : spells_list3) {
		tmpstr = !(++columns % 3) ? "\r\n" : "\t";
		out3 += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.back() == '\t')
		out += "\r\n";
	if (out2.back() == '\t')
		out2 += "\r\n";
	if (out3.back() == '\t')
		out3 += "\r\n";
	out += out2 + out3;
	return out;
}

std::string OutCasterSpellsHelp(ECharClass ch_class) {
	std::string out, out2, out3, out4;
	std::string tmpstr;
	std::vector<std::string> spells_list;
	std::vector<std::string> spells_list2;
	std::vector<std::string> spells_list3;
	std::vector<std::string> spells_list4;

	out = " :\r\n";
	out2 = "\r\n&G,      :&n\r\n";
	out3 = "\r\n&R  ,  :&n\r\n";
	out4 = "\r\n&W :&n\r\n";
	for (auto class_spell : MUD::Class(ch_class).spells) {
		int num = 0;

		if (class_spell.IsUnavailable()) {
			continue;
		}
		if (spell_create.contains(class_spell.GetId())) {
			spells_list3.push_back(MUD::Spell(class_spell.GetId()).GetCName());
		}
		if (class_spell.GetMinRemort() > 0) {
			spells_list2.push_back(MUD::Spell(class_spell.GetId()).GetCName());
			continue;
		}
		for (const auto &char_class: MUD::Classes()) {
			if (char_class.IsAvailable()) {
				for (const auto &spell : char_class.spells) {
					if (spell.GetId() == class_spell.GetId()) {
						++num;
					}
				}
			}
		}
		if (num == 1) {
			spells_list4.push_back(MUD::Spell(class_spell.GetId()).GetCName());
			continue;
		}
		spells_list.push_back(MUD::Spell(class_spell.GetId()).GetCName());
	}
	int columns = 0;

	utils::SortKoiString(spells_list);
	for (auto it : spells_list) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	columns = 0;
	utils::SortKoiString(spells_list2);
	for (auto it : spells_list2) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out2 += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	columns = 0;
	utils::SortKoiString(spells_list3);
	for (auto it : spells_list3) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out3 += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	columns = 0;
	utils::SortKoiString(spells_list4);
	for (auto it : spells_list4) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out4 += fmt::format("\t{:<30} {}", it, tmpstr);
	}
	if (out.back() == '\t')
		out += "\r\n";
	if (out2.back() == '\t')
		out2 += "\r\n";
	if (out3.back() == '\t')
		out3 += "\r\n";
	if (out4.back() == '\t')
		out4 += "\r\n";
	out += out2 + out4 + out3;
	return out;
}

void CasterSpellslHelp() {
	std::string out;

	out = OutCasterSpellsHelp(ECharClass::kSorcerer);
	out += "\r\n. : &C, , , ";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kConjurer);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kCharmer);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kWizard);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kNecromancer);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kPaladine);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutCasterSpellsHelp(ECharClass::kMerchant);
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);

	out = OutMagusSpellsHelp();
	out += "\r\n. : &C, , , &n";
	add_static("", out, 0, true);
}

std::string OutFeatureHelp(ECharClass ch_class) {
	std::stringstream out, out2, out3;
	std::string tmpstr;
	int columns = 0, columns2 = 0, columns3 = 0;
	std::vector<std::string> feat_list;
	std::vector<std::string> feat_list2;
	std::vector<std::string> feat_list3;

	out << " :\r\n";
	out2 << "\r\n&W  :&n\r\n";
	out3 << "\r\n&G :&n\r\n";
	for (const auto &feat : MUD::Class(ch_class).feats) {
		int num = 0;

		if (feat.IsUnavailable()) {
			continue;
		}
		if (feat.IsInborn()) {
			feat_list3.push_back(MUD::Feat(feat.GetId()).GetCName());
			continue;
		}
		for (const auto &char_class: MUD::Classes()) {
			if (char_class.IsAvailable()) {
				for (const auto &feat2 : char_class.feats) {
					if (feat2.GetId() == feat.GetId()) {
						++num;
					}
				}
			}
		}
		if (num == 1) {
			feat_list2.push_back(MUD::Feat(feat.GetId()).GetCName());
			continue;
		}
		feat_list.push_back(MUD::Feat(feat.GetId()).GetCName());
	}
	utils::SortKoiString(feat_list);
	for (auto it : feat_list) {
		tmpstr = !(++columns % 2) ? "\r\n" : "\t";
		out << "\t" << std::left << std::setw(30) << it << tmpstr;
	}
	utils::SortKoiString(feat_list2);
	for (auto it : feat_list2) {
		tmpstr = !(++columns2 % 2) ? "\r\n" : "\t";
		out2 << "\t" << "&C" << std::left << std::setw(30) << it << "&n" << tmpstr;
	}
	utils::SortKoiString(feat_list3);
	for (auto it : feat_list3) {
		tmpstr = !(++columns3 % 2) ? "\r\n" : "\t";
		out3 << "\t" << "&C" << std::left << std::setw(30) << it << "&n" << tmpstr;
	}
	if (out.str().back() == '\t')
		out << "\r\n";
	if (out2.str().back() == '\t')
		out2 << "\r\n";
	if (out3.str().back() == '\t')
		out3 << "\r\n";
	out << out2.str() << out3.str();
	return out.str();
}

void ClassFeatureHelp() {
	std::stringstream out;

	out << OutFeatureHelp(ECharClass::kSorcerer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kConjurer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kThief);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kWarrior);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kAssasine);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kGuard);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kCharmer);
	out << "\r\n. :, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kWizard);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kNecromancer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kPaladine);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kRanger);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kVigilant);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kMerchant);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutFeatureHelp(ECharClass::kMagus);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
}

void AllHelp() {
	std::stringstream out;

	out << OutAllSkillsHelp();
	out << "\r\n. : &C&n, &C&n, &C&n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutAllSpellsHelp();
	out << "\r\n. : &C&n, &C&n, &C&n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutAllFeaturesHelp();
	out << "\r\n. : &C&n, &C&n, &C&n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutAllRecipiesHelp();
	out << "\r\n. : &C&n, &C&n, &C&n";
	add_static("", out.str(), 0, true);
}


void ClassSkillHelp() {
	std::stringstream out;

	out << OutSkillsHelp(ECharClass::kSorcerer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kConjurer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kThief);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kWarrior);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kAssasine);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kGuard);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kCharmer);
	out << "\r\n. :, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kWizard);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kNecromancer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kPaladine);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kRanger);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kVigilant);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kMerchant);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutSkillsHelp(ECharClass::kMagus	);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
}

void ClassRecipiesHelp() {
	std::stringstream out;

	out << OutRecipiesHelp(ECharClass::kSorcerer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kConjurer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kThief);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kWarrior);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kAssasine);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kGuard);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kCharmer);
	out << "\r\n. :, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kWizard);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kNecromancer);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kPaladine);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kRanger);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kVigilant);
	out << "\r\n. : &C, , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kMerchant);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);

	out.str("");
	out << OutRecipiesHelp(ECharClass::kMagus	);
	out << "\r\n. : &C, , , &n";
	add_static("", out.str(), 0, true);
	add_static("", out.str(), 0, true);
}

void SetsHelp() {
	std::ostringstream out;
	std::string str_out;
	int count = 1;
	bool class_num[kNumPlayerClasses];

	out <<  "    ,        -     ...\r\n" <<
			"      -  () -   \r\n" <<
			" ,   !\r\n";
	table_wrapper::Table table;
	table << table_wrapper::kHeader << "#" << "" << "" << "" << table_wrapper::kEndRow;
	for (auto &it : obj_sets::sets_list) {
		int count_class = 0;

		if (!it->enabled)
			continue;
		for (auto &k : it->activ_list) {
			for (unsigned i = 0; i < kNumPlayerClasses; ++i) {
				if (k.second.prof.test(i) == true) {
					class_num[i] = true;
				} else
					class_num[i] = false;
			}
		}
		table << count++ << utils::FirstWordOnString(it->alias, " ,;") << utils::RemoveColors(it->name);
		str_out.clear();
		for (unsigned i = 0; i < kNumPlayerClasses; i++) {
			if (class_num[i]) {
				count_class++;
				str_out += MUD::Classes().FindAvailableItem(static_cast<int>(i)).GetName() + (!(count_class % 4) ? "\n" : " ");
			}
		}
		if (count_class ==  kNumPlayerClasses)
			str_out = "";
		else if (count_class == 0)
			str_out = "";
		if (str_out.back() == '\n')
			str_out.back() = '\0';
		table << str_out << table_wrapper::kSeparator << table_wrapper::kEndRow;
	}
	table.SetColumnAlign(0, table_wrapper::align::kRight);
	table_wrapper::PrintTableToStream(out, table);
	add_static("", out.str(), 0, true);
}

void init_group_zones() {
	std::stringstream out;

	for (std::size_t rnum = 0; rnum < zone_table.size(); ++rnum) {
		const auto group = zone_table[rnum].group;
		if (group > 1) {
			out << fmt::format("  {:2} - {} (. {}+).\r\n", (1 + rnum), zone_table[rnum].name, group);
		}
	}
	out << "\r\n. : &C&n";
	add_static("", out.str(), 0, true);
}

void check_update_dynamic() {
	if (need_update) {
		need_update = false;
		reload(DYNAMIC);
	}
}

void reload(Flags flag) {
	switch (flag) {
		case STATIC: static_help.clear();
			world_loader.BootIndex(DB_BOOT_HLP);
			init_group_zones();
			init_zone_all();
			ClassRecipiesHelp();
			ClassSkillHelp();
			ClassFeatureHelp();
			CasterSpellslHelp();
			SetsHelp();
			AllHelp();
			PrintActivators::process();
			obj_sets::init_xhelp();
			//      <    
//			for (auto &recode : static_help) {
//				utils::ConvertKtoW(recode.keyword);
//			}
			std::sort(static_help.begin(), static_help.end(),
					  [](const help_node &lrs, const help_node &rhs) {
						  return lrs.keyword < rhs.keyword;
					  });
//			for (auto &recode : static_help) {
//				sprintf(buf, "sort keyword=%s|", recode.keyword.c_str());
//				mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);
//			}
			break;
		case DYNAMIC: dynamic_help.clear();
			SetsDrop::init_xhelp();
			SetsDrop::init_xhelp_full();
			ClanSystem::init_xhelp();
			need_update = false;
			std::sort(dynamic_help.begin(), dynamic_help.end(),
					  [](const help_node &lrs, const help_node &rhs) {
						  return lrs.keyword < rhs.keyword;
					  });
			break;
		default: log("SYSERROR: wrong flag = %d (%s %s %d)", flag, __FILE__, __func__, __LINE__);
	};
}

void reload_all() {
	reload(STATIC);
	reload(DYNAMIC);
}

bool help_compare(const std::string &arg, const std::string &text, bool strong) {
	std::string name = arg; 

	if (strong) {

		sprintf(buf, "strong arg=%s| text=%s|",arg.c_str(), text.c_str());
		mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);
		return arg == text;
	}
	return IsEquivalent(name, text);
}

void UserSearch::process(int flag) {
	switch (flag) {
		case STATIC: search(static_help);
			break;
		case DYNAMIC: search(dynamic_help);
			break;
		default: log("SYSERROR: wrong flag = %d (%s %s %d)", flag, __FILE__, __func__, __LINE__);
	};
}

void UserSearch::print_not_found() const {
	snprintf(buf, sizeof(buf), "%s uses command HELP: %s (not found)", GET_NAME(ch), arg_str.c_str());
	mudlog(buf, LGH, kLvlImmortal, SYSLOG, true);
	snprintf(buf, sizeof(buf),
			 "&W   '&w%s&W'    .&n\r\n"
			 "\r\n&c:&n\r\n"
			 "   \"\"  ,    ,\r\n"
			 "  .       &C&n.\r\n\r\n"
			 "          .\r\n\r\n"
			 "%s",
			 arg_str.c_str(),
			 HELP_USE_EXMAPLES);
	SendMsgToChar(buf, ch);
}

void UserSearch::print_curr_topic(const help_node &node) const {
	if (node.sets_drop_page) {
		//        
		SetsDrop::print_timer_str(ch);
	}
	if (!node.no_immlog) {
		snprintf(buf, sizeof(buf), "%s uses command HELP: %s (read)",
				 GET_NAME(ch), arg_str.c_str());
		mudlog(buf, LGH, kLvlImmortal, SYSLOG, true);
	}
	page_string(ch->desc, node.entry);
}

void UserSearch::print_key_list() const {
	//   
	//      
	//          
	if (!key_list.empty() && (!diff_keys || key_list.size() == 1)) {
		print_curr_topic(*(key_list[0]));
		return;
	}
	//   
	std::stringstream out;
	out << "&W   '&w" << arg_str << "&W'    :&n\r\n\r\n";
	for (unsigned i = 0, count = 1; i < key_list.size(); ++i, ++count) {
		out << fmt::format("|&C {:<23.23} &n|", key_list[i]->keyword);
		if ((count % 3) == 0) {
			out << "\r\n";
		}
	}

	out << "\r\n\r\n"
		   "     ,    ,\r\n"
		   "     .\r\n\r\n"
		<< HELP_USE_EXMAPLES;

	snprintf(buf, sizeof(buf), "%s uses command HELP: %s (list)", GET_NAME(ch), arg_str.c_str());
	mudlog(buf, LGH, kLvlImmortal, SYSLOG, true);
	page_string(ch->desc, out.str());
}

void UserSearch::search(const std::vector<help_node> &cont) {
	//        lower_bound
	std::vector<help_node> cont1 = cont;
//   help_node     win    
	auto i = std::lower_bound(cont.begin(), cont.end(), arg_str, [](const help_node &h, const std::string& arg) {
//		sprintf(buf, "arg_str=%s| keyword=%s|",arg.c_str(), h.keyword.c_str());
//		mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);
		return h.keyword < arg;
	});
	while (i != cont.end()) {
		//       
//		sprintf(buf, "2222 arg_str=%s| keyword=%s|",arg_str.c_str(), i->keyword.c_str());
//		mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);
		if (!help_compare(arg_str, i->keyword, strong)) {
			return;
		}
//sprintf(buf, "compare");
//mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);

		//   (   )
		if (level < i->min_level) {
			++i;
			continue;
		}
//sprintf(buf, "level");
//mudlog(buf, CMP, kLvlGreatGod, SYSLOG, true);
		// key_list    ,  
		//    topic_num,  
		//      diff_keys
		for (auto & k : key_list) {
			if (k->entry != i->entry) {
				diff_keys = true;
				break;
			}
		}

		if (!topic_num) {
			key_list.push_back(i);
		} else {
			++curr_topic_num;
			//    .
			if (curr_topic_num == topic_num) {
				print_curr_topic(*i);
				stop = true;
				return;
			}
		}
		++i;
	}
}

} // namespace HelpSystem
////////////////////////////////////////////////////////////////////////////////

using namespace HelpSystem;

void do_help(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	if (!ch->desc) {
		return;
	}
	//      
	if (!*argument) {
		page_string(ch->desc, help, 0);
		return;
	}

	UserSearch user_search(ch);
	// trust_level    - kLevelImmortal
	user_search.level = GET_GOD_FLAG(ch, EGf::kDemigod) ? kLvlImmortal : GetRealLevel(ch);
	utils::ConvertToLow(argument);
	//  topic_num   
	sscanf(argument, "%d.%s", &user_search.topic_num, argument);
	//     '!' --   
	if (strlen(argument) > 1 && *(argument + strlen(argument) - 1) == '!') {
		user_search.strong = true;
		*(argument + strlen(argument) - 1) = '\0';
		user_search.arg_str = argument;
	} else {
		user_search.arg_str = utils::FixDot(argument);
	}

	//         
	for (int i = STATIC; i < TOTAL_NUM && !user_search.stop; ++i) {
		user_search.process(i);
	}
	//    ,         .
	//    -     key_list
	if (!user_search.stop) {
		if (user_search.key_list.empty()) {
			user_search.print_not_found();
		} else {
			user_search.print_key_list();
		}
	}
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
