/**
\authors Created by Sventovit
\date 2.02.2022.
\brief ,  ,           .
\details
*/

#ifndef BYLINS_SRC_STRUCTS_INFO_CONTAINER_H_
#define BYLINS_SRC_STRUCTS_INFO_CONTAINER_H_

#include <algorithm>
#include <memory>
#include <map>

#include "utils/logger.h"
#include "utils/parse.h"
#include "utils/parser_wrapper.h"

/**
 	kDisabled -   .     -     default value.
	kService -  .     ,      .
	kFrozen -   . ,   .. .
	kTesting -    .    ,        .
	kEnabled -    ,   .
 */
enum class EItemMode {
	kDisabled = 0,
	kService,
	kFrozen,
	kTesting,
	kEnabled
};

template<>
const std::string &NAME_BY_ITEM<EItemMode>(EItemMode item);
template<>
EItemMode ITEM_BY_NAME<EItemMode>(const std::string &name);

namespace info_container {

/**
 *   .   info_container     .
 */
template<typename IdEnum>
class BaseItem {
	IdEnum id_{IdEnum::kUndefined};
	EItemMode mode_{EItemMode::kDisabled};

  public:
	BaseItem() = default;
    virtual ~BaseItem() = default;
	BaseItem(IdEnum id, EItemMode mode)
	    	: id_(id), mode_(mode) {};

	[[nodiscard]] EItemMode GetMode() const { return mode_; };
	[[nodiscard]] auto GetId() const { return id_; };
	/**
	 *     .
	 */
	[[nodiscard]] bool IsValid() const { return (GetMode() > EItemMode::kFrozen); };
	/**
	 *      .
	 */
	[[nodiscard]] bool IsInvalid() const { return !IsValid(); };
	/**
	 *      ).
	 */
	[[nodiscard]] bool IsAvailable() const { return (GetMode() == EItemMode::kEnabled); };
	/**
	 *     .
	 */
	[[nodiscard]] bool IsUnavailable() const { return !IsAvailable(); };
};

template<typename Item>
class IItemBuilder {
 public:
    virtual ~IItemBuilder() = default;
	using ItemPtr = std::shared_ptr<Item>;

	virtual ItemPtr Build(parser_wrapper::DataNode &node) = 0;
	static EItemMode ParseItemMode(parser_wrapper::DataNode &node, EItemMode default_mode);
};

template<typename IdEnum, typename Item, typename ItemBuilder>
class InfoContainer {
 public:
	InfoContainer();
    virtual ~InfoContainer() = default;
	InfoContainer(InfoContainer &s) = delete;
	void operator=(const InfoContainer &s) = delete;

	using ItemPtr = std::shared_ptr<Item>;
	using Register = std::map<IdEnum, ItemPtr>;
	using NodeRange = iterators::Range<parser_wrapper::DataNode>;

/* ----------------------------------------------------------------------
 * 	 .
 ---------------------------------------------------------------------- */
	/**
	 *      id   id kUndefined.
	 */
	const Item &operator[](IdEnum id) const;
	/**
	 *  .    Reload();
	 */
	void Init(const NodeRange &data);
	/**
	 *   .    .
	 */
	void Reload(const NodeRange &data);
	/**
	 *  Id .  ,     .
	 */
	bool IsKnown(const IdEnum id) const { return items_->contains(id); };
	/**
	 *  Id .
	 */
	bool IsUnknown(const IdEnum id) const { return !IsKnown(id); };
	/**
	 *  Id   .
	 */
	bool IsValid(const IdEnum id) const { return !IsUnknown(id) && items_->at(id)->GetMode() > EItemMode::kFrozen; };
	/**
	 *  Id .
	 */
	bool IsInvalid(const IdEnum id) const { return !IsValid(id); };
	/**
	 *  Id    ).
	 */
	bool IsAvailable(const IdEnum id) const { return !IsUnknown(id) && IsEnabled(id); };
	/**
	 *  Id  (,   ).
	 */
	bool IsUnavailable(const IdEnum id) const { return !IsAvailable(id); };
	/**
	 *    .
	 */
	bool IsInitizalized() { return (items_->size() > 1); }
	/**
	 *  ,   .
	 *  const_iterator     ,
	 *    ,    .
	 */
	auto begin() const;
	auto end() const;
	/**
	 *  ,  ,    .
	 * @return -    kUndefined.
	 */
	const Item &FindItem(int num) const;
	/**
	 *   ,  ,    .
	 * @return -    kUndefined.
	 */
	const Item &FindAvailableItem(int num) const;

 private:
	friend class RegisterBuilder;
	using RegisterPtr = std::unique_ptr<Register>;

	/**
	 *      .
	 */
	class RegisterBuilder {
	 public:
		static RegisterPtr Build(const NodeRange &data, bool stop_on_error);

	 private:
		static RegisterPtr Parse(const NodeRange &data, bool stop_on_error);
		static void EmplaceItem(Register &items, ItemPtr &item);
		static void EmplaceDefaultItems(Register &items);
	};

	RegisterPtr items_;

	/**
	 *   id ,   .
	 */
	bool IsDisabled(const IdEnum id) const { return items_->at(id)->GetMode() == EItemMode::kDisabled; }
	/**
	 *   id      .
	 */
	bool IsBeingTesting(const IdEnum id) const { return items_->at(id)->GetMode() == EItemMode::kTesting; }
	/**
	 *   id    .  ,   .
	 */
	bool IsEnabled(const IdEnum id) const { return items_->at(id)->GetMode() == EItemMode::kEnabled; }

};

/* ----------------------------------------------------------------------
 * 	 InfoContainer
 ---------------------------------------------------------------------- */

template<typename IdEnum, typename Item, typename ItemBuilder>
InfoContainer<IdEnum, Item, ItemBuilder>::InfoContainer() {
	if (!items_) {
		items_ = std::make_unique<Register>();
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
void InfoContainer<IdEnum, Item, ItemBuilder>::Reload(const NodeRange &data) {
	auto new_items = RegisterBuilder::Build(data, true);
	if (new_items) {
		items_ = std::move(new_items);
	} else {
		err_log("Reloading was canceled - file damaged.");
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
void InfoContainer<IdEnum, Item, ItemBuilder>::Init(const NodeRange &data) {
	if (IsInitizalized()) {
		err_log("Don't try reinit containers. Use 'Reload()'.");
		return;
	}
	items_ = std::move(RegisterBuilder::Build(data, false));
}

template<typename IdEnum, typename Item, typename ItemBuilder>
const Item &InfoContainer<IdEnum, Item, ItemBuilder>::operator[](IdEnum id) const {
	try {
		return *(items_->at(id));
	} catch (const std::out_of_range &) {
		//err_log("Incorrect id (%d) passed into %s.", to_underlying(id), typeid(this).name()); ABYRVALG
		return *(items_->at(IdEnum::kUndefined));
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
auto InfoContainer<IdEnum, Item, ItemBuilder>::begin() const {
	iterators::ConstIterator<decltype(items_->begin()), Item> it(items_->begin());
	return it;
}

template<typename IdEnum, typename Item, typename ItemBuilder>
auto InfoContainer<IdEnum, Item, ItemBuilder>::end() const {
	iterators::ConstIterator<decltype(items_->end()), Item> it(items_->end());
	return it;
}

template<typename IdEnum, typename Item, typename ItemBuilder>
const Item &InfoContainer<IdEnum, Item, ItemBuilder>::FindItem(const int num) const {
	auto id  = static_cast<IdEnum>(num);
	if (IsKnown(id)) {
		return *(items_->at(id));
	} else {
		return *(items_->at(IdEnum::kUndefined));
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
const Item &InfoContainer<IdEnum, Item, ItemBuilder>::FindAvailableItem(const int num) const {
	auto id  = static_cast<IdEnum>(num);
	if (IsAvailable(id)) {
		return *(items_->at(id));
	} else {
		return *(items_->at(IdEnum::kUndefined));
	}
}

/* ----------------------------------------------------------------------
 * 	 RegisterBuilder
 ---------------------------------------------------------------------- */

template<typename Item>
EItemMode IItemBuilder<Item>::ParseItemMode(parser_wrapper::DataNode &node, EItemMode default_mode) {
	try {
		return parse::ReadAsConstant<EItemMode>(node.GetValue("mode"));
	} catch (std::exception &) {
		return default_mode;
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
typename InfoContainer<IdEnum, Item, ItemBuilder>::RegisterPtr
	InfoContainer<IdEnum, Item, ItemBuilder>::RegisterBuilder::Build(const NodeRange &data, bool stop_on_error) {
	auto items = Parse(data, stop_on_error);
	if (items) {
		EmplaceDefaultItems(*items);
	}
	return items;
}

template<typename IdEnum, typename Item, typename ItemBuilder>
typename InfoContainer<IdEnum, Item, ItemBuilder>::RegisterPtr
	InfoContainer<IdEnum, Item, ItemBuilder>::RegisterBuilder::Parse(const NodeRange &data, bool stop_on_error) {
	auto items = std::make_unique<Register>();

	ItemBuilder builder;
	for (auto &node : data) {
		auto item = builder.Build(node);
		if (item) {
			EmplaceItem(*items, item);
		} else if (stop_on_error) {
			return nullptr;
		}
	}

	return items;
}

template<typename IdEnum, typename Item, typename ItemBuilder>
void InfoContainer<IdEnum, Item, ItemBuilder>::RegisterBuilder::EmplaceItem(Register &items, ItemPtr &item) {
	auto id = item->GetId();
	auto it = items.try_emplace(id, std::move(item));
	if (!it.second) {
		err_log("Item '%s' has already exist. Redundant definition had been ignored.",
				NAME_BY_ITEM<IdEnum>(id).c_str());
	}
}

template<typename IdEnum, typename Item, typename ItemBuilder>
void InfoContainer<IdEnum, Item, ItemBuilder>::RegisterBuilder::EmplaceDefaultItems(Register &items) {
	auto default_item = std::make_shared<Item>();
	items.try_emplace(IdEnum::kUndefined, std::move(default_item));
	//items.try_emplace(IdEnum::kUndefined, default_item); IdEnum::kUndefined, EItemMode::kService
}

/* ----------------------------------------------------------------------
  	       
  	,      ,
  	      enum,  -   ,
  	   .

  	   -   vnum,  id  
  	  ,        ,
  	    id. __    id (   enum)
  	    .   id'  ,
  	       ,    .

  	     , 
  	  id  vnum ,    
  	  vnum.
 ---------------------------------------------------------------------- */

const int kUndefinedVnum = -1;

/**
 *        int.
 *     info_container     .
 */
template<>
class BaseItem<int> {
	int id_{-1};
	EItemMode mode_{EItemMode::kDisabled};
	std::string text_id_{"undefined"};

 public:
	BaseItem() = default;
    virtual ~BaseItem() = default;
	BaseItem(int id, std::string &text_id, EItemMode mode)
		: id_(id), mode_(mode), text_id_{text_id} {};

	[[nodiscard]] EItemMode GetMode() const { return mode_; };
	[[nodiscard]] auto GetId() const { return id_; };
	[[nodiscard]] auto GetTextId() const { return text_id_; };

	/**
	 *     .
	 */
	[[nodiscard]] bool IsValid() const { return (GetMode() > EItemMode::kFrozen); };
	/**
	 *      .
	 */
	[[nodiscard]] bool IsInvalid() const { return !IsValid(); };
	/**
	 *      ).
	 */
	[[nodiscard]] bool IsAvailable() const { return (GetMode() == EItemMode::kEnabled); };
	/**
	 *     .
	 */
	[[nodiscard]] bool IsUnavailable() const { return !IsAvailable(); };
};

template<typename Item, typename ItemBuilder>
class InfoContainer<int, Item, ItemBuilder> {
 public:
	InfoContainer();
    virtual ~InfoContainer() = default;
	InfoContainer(InfoContainer &s) = delete;
	void operator=(const InfoContainer &s) = delete;

	using ItemPtr = std::shared_ptr<Item>;
//	using Register = std::map<int, ItemPtr>;
	using NodeRange = iterators::Range<parser_wrapper::DataNode>;

/* ----------------------------------------------------------------------
 * 	      int.
 ---------------------------------------------------------------------- */
	/**
	 *      id   id kUndefined.
	 */
	const Item &operator[](int id) const;
	/**
	 *  .    Reload();
	 */
	void Init(const NodeRange &data);
	/**
	 *   .    .
	 */
	void Reload(const NodeRange &data);
	/**
	 *  Id .  ,     .
	 */
	[[nodiscard]] bool IsKnown(const int id) const { return items_->contains(id); };
	/**
	 *  Id .
	 */
	[[nodiscard]] bool IsUnknown(const int id) const { return !IsKnown(id); };
	/**
	 *  Id   .
	 */
	[[nodiscard]] bool IsValid(const int id) const {
      return !IsUnknown(id) && items_->at(id)->GetMode() > EItemMode::kFrozen;
    };
	/**
	 *  Id .
	 */
	[[nodiscard]] bool IsInvalid(const int id) const { return !IsValid(id); };
	/**
	 *  Id    ).
	 */
	[[nodiscard]] bool IsAvailable(const int id) const { return !IsUnknown(id) && IsEnabled(id); };
	/**
	 *  Id  (,   ).
	 */
	[[nodiscard]] bool IsUnavailable(const int id) const { return !IsAvailable(id); };
	/**
	 *    .
	 */
	[[nodiscard]] bool IsInitizalized() { return (items_->size() > 1); }
	/**
	 *  ,   .
	 *  const_iterator     ,
	 *    ,    .
	 */
	auto begin() const;
	auto end() const;
	/**
	 *  ,    id.
	 * @return -    kUndefined.
	 */
	[[nodiscard]] const Item &FindItem(const std::string &text_id) const;
	/**
	 *   ,    id.
	 * @return -    kUndefined.
	 */
	[[nodiscard]] const Item &FindAvailableItem(const std::string &text_id) const;

 private:
	friend class RegisterBuilder;
	using Register = std::map<int, ItemPtr>;
	using TextIdRegister = std::unordered_map<std::string, ItemPtr>;
	using RegisterPtr = std::unique_ptr<Register>;
	using TextIdRegisterPtr = std::unique_ptr<TextIdRegister>;

	/**
	 *      .
	 */
	class RegisterBuilder {
	 public:
		static RegisterPtr Build(const NodeRange &data, bool stop_on_error);

	 private:
		static RegisterPtr Parse(const NodeRange &data, bool stop_on_error);
		static void EmplaceItem(Register &items, ItemPtr &item);
		static void EmplaceDefaultItems(Register &items);
	};

	RegisterPtr items_;
	TextIdRegisterPtr text_ids_register_;

	/**
	 *   id ,   .
	 */
	[[nodiscard]] bool IsDisabled(const int id) const { return items_->at(id)->GetMode() == EItemMode::kDisabled; }
	/**
	 *   id      .
	 */
	[[nodiscard]] bool IsBeingTesting(const int id) const { return items_->at(id)->GetMode() == EItemMode::kTesting; }
	/**
	 *   id    .  ,   .
	 */
	[[nodiscard]] bool IsEnabled(const int id) const { return items_->at(id)->GetMode() == EItemMode::kEnabled; }

	void BuildTextIdsRegister();
};

/* ----------------------------------------------------------------------
 * 	 InfoContainer     int
 ---------------------------------------------------------------------- */

template<typename Item, typename ItemBuilder>
InfoContainer<int, Item, ItemBuilder>::InfoContainer() {
	if (!items_) {
		items_ = std::make_unique<Register>();
	}
	if (!text_ids_register_) {
		text_ids_register_ = std::make_unique<TextIdRegister>();
	}
}

template<typename Item, typename ItemBuilder>
void InfoContainer<int, Item, ItemBuilder>::Reload(const NodeRange &data) {
	auto new_items = RegisterBuilder::Build(data, true);
	if (new_items) {
		items_ = std::move(new_items);
		BuildTextIdsRegister();
	} else {
		err_log("Reloading was canceled - file damaged.");
	}
}

template<typename Item, typename ItemBuilder>
void InfoContainer<int, Item, ItemBuilder>::Init(const NodeRange &data) {
	if (IsInitizalized()) {
		err_log("Don't try reinit containers. Use 'Reload()'.");
		return;
	}
	items_ = std::move(RegisterBuilder::Build(data, false));
	BuildTextIdsRegister();
}

template<typename Item, typename ItemBuilder>
void InfoContainer<int, Item, ItemBuilder>::BuildTextIdsRegister() {
	text_ids_register_->clear();
	for (const auto &[key, val] : *items_) {
		text_ids_register_->try_emplace(val->GetTextId(), val);
	}
}

template<typename Item, typename ItemBuilder>
const Item &InfoContainer<int, Item, ItemBuilder>::operator[](int id) const {
	try {
		return *(items_->at(id));
	} catch (const std::out_of_range &) {
		return *(items_->at(kUndefinedVnum));
	}
}

template<typename Item, typename ItemBuilder>
auto InfoContainer<int, Item, ItemBuilder>::begin() const {
	iterators::ConstIterator<decltype(items_->begin()), Item> it(items_->begin());
	return it;
}

template<typename Item, typename ItemBuilder>
auto InfoContainer<int, Item, ItemBuilder>::end() const {
	iterators::ConstIterator<decltype(items_->end()), Item> it(items_->end());
	return it;
}

template<typename Item, typename ItemBuilder>
const Item &InfoContainer<int, Item, ItemBuilder>::FindItem(const std::string &text_id) const {
	auto it = text_ids_register_->find(text_id);
	if (it != text_ids_register_->end()) {
		return *(it->second);
	} else {
		return *(items_->at(kUndefinedVnum));
	}
}

template<typename Item, typename ItemBuilder>
const Item &InfoContainer<int, Item, ItemBuilder>::FindAvailableItem(const std::string &text_id) const {
	auto it = text_ids_register_->find(text_id);
	if (it != text_ids_register_->end() && it->second->IsAvailable()) {
		return *(it->second);
	} else {
		return *(items_->at(kUndefinedVnum));
	}
}

/* ----------------------------------------------------------------------
 * 	 RegisterBuilder     int
 ---------------------------------------------------------------------- */

template<typename Item, typename ItemBuilder>
typename InfoContainer<int, Item, ItemBuilder>::RegisterPtr
InfoContainer<int, Item, ItemBuilder>::RegisterBuilder::Build(const NodeRange &data, bool stop_on_error) {
	auto items = Parse(data, stop_on_error);
	if (items) {
		EmplaceDefaultItems(*items);
	}
	return items;
}

template<typename Item, typename ItemBuilder>
typename InfoContainer<int, Item, ItemBuilder>::RegisterPtr
InfoContainer<int, Item, ItemBuilder>::RegisterBuilder::Parse(const NodeRange &data, bool stop_on_error) {
	auto items = std::make_unique<Register>();

	ItemBuilder builder;
	for (auto &node : data) {
		auto item = builder.Build(node);
		if (item) {
			EmplaceItem(*items, item);
		} else if (stop_on_error) {
			return nullptr;
		}
	}

	return items;
}

template<typename Item, typename ItemBuilder>
void InfoContainer<int, Item, ItemBuilder>::RegisterBuilder::EmplaceItem(Register &items, ItemPtr &item) {
	auto id = item->GetId();
	auto it = items.try_emplace(id, std::move(item));
	if (!it.second) {
		err_log("Item with vnum '%d' has already exist. Redundant definition had been ignored.", id);
	}
}

template<typename Item, typename ItemBuilder>
void InfoContainer<int, Item, ItemBuilder>::RegisterBuilder::EmplaceDefaultItems(Register &items) {
	auto default_item = std::make_shared<Item>();
	items.try_emplace(kUndefinedVnum, std::move(default_item));
}


} // info_container

#endif //BYLINS_SRC_STRUCTS_INFO_CONTAINER_H_

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
