/**
\file do_steal.cpp - a part of the Bylins engine.
\authors Created by Sventovit.
\date 17.09.2024.
\brief Brief description.
\detail Detail description.
*/

#include "engine/entities/char_data.h"
#include "engine/entities/obj_data.h"
#include "engine/core/handler.h"
#include "gameplay/skills/skills.h"
#include "engine/db/global_objects.h"
#include "gameplay/core/base_stats.h"
#include "gameplay/fight/fight_constants.h"
#include "gameplay/fight/fight_hit.h"
#include "do_get.h"
#include "engine/core/utils_char_obj.inl"
#include "gameplay/ai/spec_procs.h"

void go_steal(CharData *ch, CharData *vict, char *obj_name) {
	int percent, gold, eq_pos, ohoh = 0, success = 0, prob;
	ObjData *obj;

	if (!vict) {
		return;
	}

	if (!IS_IMMORTAL(ch) && vict->GetEnemy()) {
		act("$N   .", false, ch, nullptr, vict, kToChar);
		return;
	}

	if (!IS_IMMORTAL(ch) && ROOM_FLAGGED(vict->in_room, ERoomFlag::kArena)) {
		SendMsgToChar("   .\r\n", ch);
		return;
	}

	// 101% is a complete failure
	percent = number(1, MUD::Skill(ESkill::kSteal).difficulty);

	if (IS_IMMORTAL(ch) || (vict->GetPosition() <= EPosition::kSleep && !AFF_FLAGGED(vict, EAffect::kSleep)))
		success = 1;    // ALWAYS SUCCESS, unless heavy object.

	if (!AWAKE(vict))    // Easier to steal from sleeping people.
		percent = MAX(percent - 50, 0);

	// NO, NO With Imp's and Shopkeepers, and if player thieving is not allowed
	if ((IS_IMMORTAL(vict) || GET_GOD_FLAG(vict, EGf::kGodsLike) || GET_MOB_SPEC(vict) == shop_ext)
		&& !IS_IMPL(ch)) {
		SendMsgToChar("      .\r\n", ch);
		return;
	}

	if (str_cmp(obj_name, "coins")
		&& str_cmp(obj_name, "gold")
		&& str_cmp(obj_name, "")
		&& str_cmp(obj_name, "")) {
		if (!(obj = get_obj_in_list_vis(ch, obj_name, vict->carrying))) {
			for (eq_pos = 0; eq_pos < EEquipPos::kNumEquipPos; eq_pos++) {
				if (GET_EQ(vict, eq_pos)
					&& (isname(obj_name, GET_EQ(vict, eq_pos)->get_aliases()))
					&& CAN_SEE_OBJ(ch, GET_EQ(vict, eq_pos))) {
					obj = GET_EQ(vict, eq_pos);
					break;
				}
			}
			if (!obj) {
				act("  $S    - -- (2 )...", false, ch, nullptr, vict, kToChar);
				return;
			} else    // It is equipment
			{
				if (!success) {
					SendMsgToChar("?  ? --!\r\n", ch);
					return;
				} else if (ch->GetCarryingQuantity() >= CAN_CARRY_N(ch)) {
					SendMsgToChar("     .\r\n", ch);
					return;
				} else if (ch->GetCarryingWeight() + GET_OBJ_WEIGHT(obj) > CAN_CARRY_W(ch)) {
					SendMsgToChar("     .\r\n", ch);
					return;
				} else if (obj->has_flag(EObjFlag::kBloody)) {
					SendMsgToChar(
						"\" !\" -     ,      ,    .\r\n",
						ch);
					return;
				} else {
					act("  $N3   $o3.", false, ch, obj, vict, kToChar);
					act("$n $g $o3  $N1.", false, ch, obj, vict, kToNotVict | kToArenaListen);
					PlaceObjToInventory(UnequipChar(vict, eq_pos, CharEquipFlags()), ch);
				}
			}
		} else    // obj found in inventory
		{
			if (obj->has_flag(EObjFlag::kBloody)) {
				SendMsgToChar(
					"\" !\" -     ,      ,    .\r\n",
					ch);
				return;
			}
			percent += GET_OBJ_WEIGHT(obj);    // Make heavy harder
			prob = CalcCurrentSkill(ch, ESkill::kSteal, vict);

			if (AFF_FLAGGED(ch, EAffect::kHide))
				prob += 5;
			if (!IS_IMMORTAL(ch) && AFF_FLAGGED(vict, EAffect::kSleep))
				prob = 0;
			if (percent > prob && !success) {
				ohoh = true;
				if (AFF_FLAGGED(ch, EAffect::kHide)) {
					RemoveAffectFromChar(ch, ESpell::kHide);
					AFF_FLAGS(ch).unset(EAffect::kHide);
					SendMsgToChar("  .\r\n", ch);
					act("$n $g .", false, ch, nullptr, nullptr, kToRoom);
				};
				SendMsgToChar("..   !\r\n", ch);
				act("$n $u  !", false, ch, nullptr, vict, kToVict);
				act("$n $u    $N1.", true, ch, nullptr, vict, kToNotVict | kToArenaListen);
			} else    // Steal the item
			{
				if (ch->GetCarryingQuantity() + 1 < CAN_CARRY_N(ch)) {
					if (ch->GetCarryingWeight() + GET_OBJ_WEIGHT(obj) < CAN_CARRY_W(ch)) {
						RemoveObjFromChar(obj);
						PlaceObjToInventory(obj, ch);
						act("  $o3  $N1!", false, ch, obj, vict, kToChar);
					}
				} else {
					SendMsgToChar("    .\r\n", ch);
					return;
				}
			}
			if (CAN_SEE(vict, ch) && AWAKE(vict))
				ImproveSkill(ch, ESkill::kSteal, 0, vict);
		}
	} else        // Steal some coins
	{
		prob = CalcCurrentSkill(ch, ESkill::kSteal, vict);
		if (AFF_FLAGGED(ch, EAffect::kHide))
			prob += 5;
		if (!IS_IMMORTAL(ch) && AFF_FLAGGED(vict, EAffect::kSleep))
			prob = 0;
		if (percent > prob && !success) {
			ohoh = true;
			if (AFF_FLAGGED(ch, EAffect::kHide)) {
				RemoveAffectFromChar(ch, ESpell::kHide);
				AFF_FLAGS(ch).unset(EAffect::kHide);
				SendMsgToChar("  .\r\n", ch);
				act("$n $g .", false, ch, nullptr, nullptr, kToRoom);
			};
			SendMsgToChar(" ...  ...    ..\r\n", ch);
			act("   $n1   .", false, ch, nullptr, vict, kToVict);
			act("$n $u    $N1.", true, ch, nullptr, vict, kToNotVict | kToArenaListen);
		} else    // Steal some gold coins
		{
			if (!vict->get_gold()) {
				act("$E $A,    :)", false, ch, nullptr, vict, kToChar);
				return;
			} else {
				//   - (  )
				if ((number(1, 100) - ch->GetSkill(ESkill::kSteal) -
					ch->get_dex() + vict->get_wis() + vict->get_gold() / 500) < 0) {
					act("  $N1   .", true, ch, nullptr, vict, kToChar);
					gold = vict->get_gold();
				} else
					gold = (int) ((vict->get_gold() * number(1, 75)) / 100);

				if (gold > 0) {
					if (gold > 1) {
						sprintf(buf, "---!    %d %s.\r\n",
								gold, GetDeclensionInNumber(gold, EWhat::kMoneyU));
						SendMsgToChar(buf, ch);
					} else {
						SendMsgToChar("-- !   :) 1 ()  :(.\r\n", ch);
					}
					ch->add_gold(gold);
					sprintf(buf,
							"<%s> {%d}   %d   %s.",
							ch->get_name().c_str(),
							GET_ROOM_VNUM(ch->in_room),
							gold,
							GET_PAD(vict, 0));
					mudlog(buf, NRM, kLvlGreatGod, MONEY_LOG, true);
					split_or_clan_tax(ch, gold);
					vict->remove_gold(gold);
				} else
					SendMsgToChar("    ...\r\n", ch);
			}
		}
		if (CAN_SEE(vict, ch) && AWAKE(vict))
			ImproveSkill(ch, ESkill::kSteal, 0, vict);
	}
	if (!IS_IMMORTAL(ch) && ohoh)
		SetWaitState(ch, 3 * kBattleRound);
	pk_thiefs_action(ch, vict);
	if (ohoh && vict->IsNpc() && AWAKE(vict) && CAN_SEE(vict, ch) && MAY_ATTACK(vict))
		hit(vict, ch, ESkill::kUndefined, fight::kMainHand);
}

void do_steal(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	CharData *vict;
	char vict_name[kMaxInputLength], obj_name[kMaxInputLength];

	if (ch->IsNpc() || !ch->GetSkill(ESkill::kSteal)) {
		SendMsgToChar("    .\r\n", ch);
		return;
	}
	if (!IS_IMMORTAL(ch) && ch->IsOnHorse()) {
		SendMsgToChar("   .\r\n", ch);
		return;
	}
	two_arguments(argument, obj_name, vict_name);
	if (!(vict = get_char_vis(ch, vict_name, EFind::kCharInRoom))) {
		SendMsgToChar("  ?\r\n", ch);
		return;
	} else if (vict == ch) {
		SendMsgToChar("  \" <n> \".\r\n", ch);
		return;
	}
	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kPeaceful) && !(IS_IMMORTAL(ch) || GET_GOD_FLAG(ch, EGf::kGodsLike))) {
		SendMsgToChar("  .      ...\r\n", ch);
		return;
	}
	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kHouse) && !vict->IsNpc()) {
		SendMsgToChar("  ?  ...\r\n", ch);
		return;
	}
	if (vict->IsNpc() && (vict->IsFlagged(EMobFlag::kNoFight) || AFF_FLAGGED(vict, EAffect::kGodsShield)
		|| vict->IsFlagged(EMobFlag::kProtect))
		&& !(IS_IMMORTAL(ch) || GET_GOD_FLAG(ch, EGf::kGodsLike))) {
		SendMsgToChar("  ?  !\r\n ,     .\r\n", ch);
		return;
	}
	go_steal(ch, vict, obj_name);
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
