/*************************************************************************
*   File: act.comm.cpp                                  Part of Bylins    *
*  Usage: Player-level communication commands                             *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
*                                                                         *
*  $Author$                                                        *
*  $Date$                                           *
*  $Revision$                                                       *
************************************************************************ */

#include "handler.h"
#include "color.h"
#include "game_economics/auction.h"
#include "entities/char_player.h"
#include "entities/world_characters.h"
#include "house.h"
#include "spam.h"
//#include "stuff.h"
#include "utils/utils_char_obj.inl"

// extern variables
/*extern DescriptorData *descriptor_list;
extern TimeInfoData time_info;*/

// local functions
void perform_tell(CharData *ch, CharData *vict, char *arg);
int is_tell_ok(CharData *ch, CharData *vict);
bool tell_can_see(CharData *ch, CharData *vict);

// external functions
extern char *diag_timer_to_char(const ObjData *obj);
extern void SetWait(CharData *ch, int waittime, int victim_in_room);

void do_say(CharData *ch, char *argument, int cmd, int subcmd);
void do_gsay(CharData *ch, char *argument, int cmd, int subcmd);
void do_tell(CharData *ch, char *argument, int cmd, int subcmd);
void do_reply(CharData *ch, char *argument, int cmd, int subcmd);
void do_spec_comm(CharData *ch, char *argument, int cmd, int subcmd);
void do_write(CharData *ch, char *argument, int cmd, int subcmd);
void do_page(CharData *ch, char *argument, int cmd, int subcmd);
void do_gen_comm(CharData *ch, char *argument, int cmd, int subcmd);
void do_pray_gods(CharData *ch, char *argument, int cmd, int subcmd);
void do_ignore(CharData *ch, char *argument, int cmd, int subcmd);

#define SIELENCE (" ,    .\r\n")
#define SOUNDPROOF ("   .\r\n")

void do_say(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {

//	create_charmice_stuff(ch, static_cast<ESkill>(atoi(argument)), 100);//    

	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}

	if (!*argument)
		SendMsgToChar(" : \"   ?\"\r\n", ch);
	else {
		sprintf(buf, "$n $g : '%s'", argument);

//      
//   act      
		for (const auto to : world[ch->in_room]->people) {
			if (ch == to || ignores(to, ch, EIgnore::kSay)) {
				continue;
			}
			act(buf, false, ch, 0, to, kToVict | DG_NO_TRIG | kToNotDeaf);
		}

		if (!ch->IsNpc() && PRF_FLAGGED(ch, EPrf::kNoRepeat)) {
			SendMsgToChar(OK, ch);
		} else {
			delete_doubledollar(argument);
			sprintf(buf, "  : '%s'\r\n", argument);
			SendMsgToChar(buf, ch);
		}
		speech_mtrigger(ch, argument);
		speech_wtrigger(ch, argument);
	}
}

void do_gsay(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	CharData *k;
	struct FollowerType *f;

	if (AFF_FLAGGED(ch, EAffect::kSilence)
		|| AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}

	skip_spaces(&argument);

	if (!AFF_FLAGGED(ch, EAffect::kGroup)) {
		SendMsgToChar("    !\r\n", ch);
		return;
	}

	if (!*argument) {
		SendMsgToChar("      ?\r\n", ch);
	} else {
		if (ch->has_master()) {
			k = ch->get_master();
		} else {
			k = ch;
		}

		sprintf(buf, "$n $g  : '%s'", argument);

		if (AFF_FLAGGED(k, EAffect::kGroup)
			&& k != ch
			&& !ignores(k, ch, EIgnore::kGroup)) {
			act(buf, false, ch, 0, k, kToVict | kToSleep | kToNotDeaf);
			// added by WorM   2010.10.13
			if (!AFF_FLAGGED(k, EAffect::kDeafness)
				&& GET_POS(k) > EPosition::kDead) {
				sprintf(buf1,
						"%s %s  : '%s'\r\n",
						tell_can_see(ch, k) ? GET_NAME(ch) : "-",
						GET_CH_VIS_SUF_1(ch, k),
						argument);
				k->remember_add(buf1, Remember::ALL);
				k->remember_add(buf1, Remember::GROUP);
			}
			//end by WorM
		}
		for (f = k->followers; f; f = f->next) {
			if (AFF_FLAGGED(f->follower, EAffect::kGroup)
				&& (f->follower != ch)
				&& !ignores(f->follower, ch, EIgnore::kGroup)) {
				act(buf, false, ch, 0, f->follower, kToVict | kToSleep | kToNotDeaf);
				// added by WorM   2010.10.13
				if (!AFF_FLAGGED(f->follower, EAffect::kDeafness)
					&& GET_POS(f->follower) > EPosition::kDead) {
					sprintf(buf1,
							"%s %s  : '%s'\r\n",
							tell_can_see(ch, f->follower) ? GET_NAME(ch) : "-",
							GET_CH_VIS_SUF_1(ch, f->follower),
							argument);
					f->follower->remember_add(buf1, Remember::ALL);
					f->follower->remember_add(buf1, Remember::GROUP);
				}
				//end by WorM
			}
		}

		if (PRF_FLAGGED(ch, EPrf::kNoRepeat))
			SendMsgToChar(OK, ch);
		else {
			sprintf(buf, "   : '%s'\r\n", argument);
			SendMsgToChar(buf, ch);
			// added by WorM   2010.10.13
			ch->remember_add(buf, Remember::ALL);
			ch->remember_add(buf, Remember::GROUP);
			//end by WorM
		}
	}
}

bool tell_can_see(CharData *ch, CharData *vict) {
	if (CAN_SEE_CHAR(vict, ch) || IS_IMMORTAL(ch) || GET_INVIS_LEV(ch)) {
		return true;
	} else {
		return false;
	}
}

void perform_tell(CharData *ch, CharData *vict, char *arg) {
// shapirus:   ,      
//  ;    
	if (PRF_FLAGGED(vict, EPrf::kNoInvistell)
		&& !CAN_SEE(vict, ch)
		&& GetRealLevel(ch) < kLvlImmortal
		&& !PRF_FLAGGED(ch, EPrf::kCoderinfo)) {
		act("$N     ,   .", false, ch, 0, vict, kToChar | kToSleep);
		return;
	}

	// TODO:   act()   ,        act()
	if (tell_can_see(ch, vict)) {
		snprintf(buf, kMaxStringLength, "%s %s  : '%s'", GET_NAME(ch), GET_CH_SUF_1(ch), arg);
	} else {
		snprintf(buf, kMaxStringLength, "-   : '%s'", arg);
	}
	snprintf(buf1, kMaxStringLength, "%s%s%s\r\n", CCICYN(vict, C_NRM), CAP(buf), CCNRM(vict, C_NRM));
	SendMsgToChar(buf1, vict);
	if (!vict->IsNpc()) {
		vict->remember_add(buf1, Remember::ALL);
	}

	if (!vict->IsNpc() && !ch->IsNpc()) {
		snprintf(buf, kMaxStringLength, "%s%s : '%s'%s\r\n", CCICYN(vict, C_NRM),
				 tell_can_see(ch, vict) ? GET_NAME(ch) : "-", arg, CCNRM(vict, C_NRM));
		vict->remember_add(buf, Remember::PERSONAL);
	}

	if (!ch->IsNpc() && PRF_FLAGGED(ch, EPrf::kNoRepeat)) {
		SendMsgToChar(OK, ch);
	} else {
		snprintf(buf, kMaxStringLength, "%s  %s : '%s'%s\r\n", CCICYN(ch, C_NRM),
				 tell_can_see(vict, ch) ? vict->player_data.PNames[2].c_str() : "-", arg, CCNRM(ch, C_NRM));
		SendMsgToChar(buf, ch);
		if (!ch->IsNpc()) {
			ch->remember_add(buf, Remember::ALL);
		}
	}

	if (!vict->IsNpc() && !ch->IsNpc()) {
		vict->set_answer_id(GET_IDNUM(ch));
	}
}

int is_tell_ok(CharData *ch, CharData *vict) {
	if (ch == vict) {
		SendMsgToChar("      .\r\n", ch);
		return (false);
	} else if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     .\r\n", ch);
		return (false);
	} else if (!vict->IsNpc() && !vict->desc)    // linkless
	{
		act("$N $G    .", false, ch, 0, vict, kToChar | kToSleep);
		return (false);
	} else if (PLR_FLAGGED(vict, EPlrFlag::kWriting)) {
		act("$N   -  .", false, ch, 0, vict, kToChar | kToSleep);
		return (false);
	}

	if (IS_GOD(ch) || PRF_FLAGGED(ch, EPrf::kCoderinfo))
		return (true);

	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kSoundproof))
		SendMsgToChar(SOUNDPROOF, ch);
	else if ((!vict->IsNpc() &&
		(PRF_FLAGGED(vict, EPrf::kNoTell) || ignores(vict, ch, EIgnore::kTell))) ||
		ROOM_FLAGGED(vict->in_room, ERoomFlag::kSoundproof)) {
		if (ch->IsNpc()) {
			if (ROOM_FLAGGED(vict->in_room, ERoomFlag::kSoundproof)) {
				log("NOTELL: name %s room %d soundproof", GET_NAME(vict), GET_ROOM_VNUM(vict->in_room));
			}
			if (PRF_FLAGGED(vict, EPrf::kNoTell))
				log("NOTELL: name %s   ", GET_NAME(vict));
		}
		act("$N    .", false, ch, 0, vict, kToChar | kToSleep);
	}
	else if (GET_POS(vict) < EPosition::kRest || AFF_FLAGGED(vict, EAffect::kDeafness))
		act("$N   .", false, ch, 0, vict, kToChar | kToSleep);
	else
		return (true);

	return (false);
}

/*
 * Yes, do_tell probably could be combined with whisper and ask, but
 * called frequently, and should IMHO be kept as tight as possible.
 */
void do_tell(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	CharData *vict = nullptr;

	if (AFF_FLAGGED(ch, EAffect::kCharmed))
		return;

	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	/*   
	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kTribune))
	{
		SendMsgToChar(SOUNDPROOF, ch);
		return;
	}
	*/

	half_chop(argument, buf, buf2);

	if (!*buf || !*buf2) {
		SendMsgToChar("     ?\r\n", ch);
	} else if (!(vict = get_player_vis(ch, buf, EFind::kCharInWorld))) {
		SendMsgToChar(NOPERSON, ch);
	} else if (vict->IsNpc())
		SendMsgToChar(NOPERSON, ch);
	else if (is_tell_ok(ch, vict)) {
		if (PRF_FLAGGED(ch, EPrf::kNoTell))
			SendMsgToChar("   !\r\n", ch);
		perform_tell(ch, vict, buf2);
	}
}

void do_reply(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	if (ch->IsNpc())
		return;

	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}

	skip_spaces(&argument);

	if (ch->get_answer_id() == kNobody)
		SendMsgToChar("  !\r\n", ch);
	else if (!*argument)
		SendMsgToChar("   ?\r\n", ch);
	else {
		/*
		 * Make sure the person you're replying to is still playing by searching
		 * for them.  Note, now last tell is stored as player IDnum instead of
		 * a pointer, which is much better because it's safer, plus will still
		 * work if someone logs out and back in again.
		 */

		/*
		 * XXX: A descriptor list based search would be faster although
		 *      we could not find link dead people.  Not that they can
		 *      hear tells anyway. :) -gg 2/24/98
		 */
		bool found = false;
		for (const auto &i : character_list) {
			if (!i->IsNpc()
				&& GET_IDNUM(i) == ch->get_answer_id()) {
				if (is_tell_ok(ch, i.get())) {
					perform_tell(ch, i.get(), argument);
				}

				found = true;

				break;
			}
		}

		if (!found) {
			SendMsgToChar("     .\r\n", ch);
		}
	}
}

void do_spec_comm(CharData *ch, char *argument, int/* cmd*/, int subcmd) {
	CharData *vict;
	const char *action_sing, *action_plur, *action_others, *vict1, *vict2;
	char vict3[kMaxInputLength];

	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}

	if (subcmd == SCMD_WHISPER) {
		action_sing = "";
		vict1 = "";
		vict2 = "";
		action_plur = "";
		action_others = "$n - $g $N2.";
	} else {
		action_sing = "";
		vict1 = " ";
		vict2 = " ";
		action_plur = "";
		action_others = "$n $g $N2 .";
	}

	half_chop(argument, buf, buf2);

	if (!*buf || !*buf2) {
		sprintf(buf, "   %s..  %s?\r\n", action_sing, vict1);
		SendMsgToChar(buf, ch);
	} else if (!(vict = get_char_vis(ch, buf, EFind::kCharInRoom)))
		SendMsgToChar(NOPERSON, ch);
	else if (vict == ch)
		SendMsgToChar("     -   ...\r\n", ch);
	else if (ignores(vict, ch, subcmd == SCMD_WHISPER ? EIgnore::kWhisper : EIgnore::kAsk)) {
		sprintf(buf, "%s    .\r\n", GET_NAME(vict));
		SendMsgToChar(buf, ch);
	} else {
		if (subcmd == SCMD_WHISPER)
			sprintf(vict3, "%s", GET_PAD(vict, 2));
		else
			sprintf(vict3, " %s", GET_PAD(vict, 1));

		std::stringstream buffer;
		buffer << "$n " << action_plur << "$g " << vict2 << " : " << buf2;
//		sprintf(buf, "$n %s$g %s : '%s'", action_plur, vict2, buf2);
		act(buffer.str().c_str(), false, ch, 0, vict, kToVict | kToNotDeaf);

		if (PRF_FLAGGED(ch, EPrf::kNoRepeat))
			SendMsgToChar(OK, ch);
		else {
			std::stringstream buffer;
			buffer << " " << action_plur << " " << vict3 << " : '" << buf2 << "'" << std::endl;
//			sprintf(buf, " %s %s : '%s'\r\n", action_plur, vict3, buf2);
			SendMsgToChar(buffer.str(), ch);
		}

		act(action_others, false, ch, 0, vict, kToNotVict);
	}
}

#define MAX_NOTE_LENGTH 4096    // arbitrary

void do_write(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	ObjData *paper, *pen = nullptr;
	char papername[kMaxInputLength], penname[kMaxInputLength];

	two_arguments(argument, papername, penname);

	if (!ch->desc)
		return;

	if (!*papername)    // nothing was delivered
	{
		SendMsgToChar("?  ?    ?\r\n", ch);
		return;
	}
	if (*penname)        // there were two arguments
	{
		if (!(paper = get_obj_in_list_vis(ch, papername, ch->carrying))) {
			sprintf(buf, "   %s.\r\n", papername);
			SendMsgToChar(buf, ch);
			return;
		}
		if (!(pen = get_obj_in_list_vis(ch, penname, ch->carrying))) {
			sprintf(buf, "   %s.\r\n", penname);
			SendMsgToChar(buf, ch);
			return;
		}
	} else        // there was one arg.. let's see what we can find
	{
		if (!(paper = get_obj_in_list_vis(ch, papername, ch->carrying))) {
			sprintf(buf, "   %s  .\r\n", papername);
			SendMsgToChar(buf, ch);
			return;
		}
		if (GET_OBJ_TYPE(paper) == EObjType::kPen)    // oops, a pen..
		{
			pen = paper;
			paper = nullptr;
		} else if (GET_OBJ_TYPE(paper) != EObjType::kNote) {
			SendMsgToChar("     .\r\n", ch);
			return;
		}
		// One object was found.. now for the other one.
		if (!GET_EQ(ch, kHold)) {
			sprintf(buf, "  !\r\n");
			SendMsgToChar(buf, ch);
			return;
		}
		if (!CAN_SEE_OBJ(ch, GET_EQ(ch, kHold))) {
			SendMsgToChar("  - !  ,    !!\r\n", ch);
			return;
		}
		if (pen)
			paper = GET_EQ(ch, kHold);
		else
			pen = GET_EQ(ch, kHold);
	}


	// ok.. now let's see what kind of stuff we've found
	if (GET_OBJ_TYPE(pen) != EObjType::kPen) {
		act("    $o4.", false, ch, pen, 0, kToChar);
	} else if (GET_OBJ_TYPE(paper) != EObjType::kNote) {
		act("     $o5.", false, ch, paper, 0, kToChar);
	} else if (!paper->get_action_description().empty()) {
		SendMsgToChar("  - .\r\n", ch);
	} else            // we can write - hooray!
	{
		/* this is the PERFECT code example of how to set up:
		 * a) the text editor with a message already loaed
		 * b) the abort buffer if the player aborts the message
		 */
		ch->desc->backstr = nullptr;
		SendMsgToChar(" .  (/s    /h )\r\n", ch);
		// ok, here we check for a message ALREADY on the paper
		if (!paper->get_action_description().empty())    // we str_dup the original text to the descriptors->backstr
		{
			ch->desc->backstr = str_dup(paper->get_action_description().c_str());
			// send to the player what was on the paper (cause this is already
			// loaded into the editor)
			SendMsgToChar(paper->get_action_description().c_str(), ch);
		}
		act("$n $g .", true, ch, 0, 0, kToRoom);
		// assign the descriptor's->str the value of the pointer to the text
		// pointer so that we can reallocate as needed (hopefully that made
		// sense :>)
		const utils::AbstractStringWriter::shared_ptr writer(new CActionDescriptionWriter(*paper));
		string_write(ch->desc, writer, MAX_NOTE_LENGTH, 0, nullptr);
	}
}

void do_page(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	DescriptorData *d;
	CharData *vict;

	half_chop(argument, arg, buf2);

	if (ch->IsNpc())
		SendMsgToChar("--   .. .\r\n", ch);
	else if (!*arg)
		SendMsgToChar("Whom do you wish to page?\r\n", ch);
	else {
		std::stringstream buffer;
		buffer << "\007\007*$n*" << buf2;
//		sprintf(buf, "\007\007*$n* %s", buf2);
		if (!str_cmp(arg, "all") || !str_cmp(arg, "")) {
			if (IS_GRGOD(ch)) {
				for (d = descriptor_list; d; d = d->next) {
					if (STATE(d) == CON_PLAYING && d->character) {
						act(buf, false, ch, 0, d->character.get(), kToVict);
					}
				}
			} else {
				SendMsgToChar("   !\r\n", ch);
			}
			return;
		}
		if ((vict = get_char_vis(ch, arg, EFind::kCharInWorld)) != nullptr) {
			act(buffer.str().c_str(), false, ch, 0, vict, kToVict);
			if (PRF_FLAGGED(ch, EPrf::kNoRepeat))
				SendMsgToChar(OK, ch);
			else
				act(buffer.str().c_str(), false, ch, 0, vict, kToChar);
		} else
			SendMsgToChar("  !\r\n", ch);
	}
}

/**********************************************************************
 * generalized communication func, originally by Fred C. Merkel (Torg) *
  *********************************************************************/

struct communication_type {
	const char *muted_msg;
	const char *action;
	const char *no_channel;
	const char *color;
	const char *you_action;
	const char *hi_action;
	int min_lev;
	int move_cost;
	int noflag;
};

void do_gen_comm(CharData *ch, char *argument, int/* cmd*/, int subcmd) {
	DescriptorData *i;
	char color_on[24];
	int ign_flag;
	/*
	 * com_msgs: Message if you can't perform the action because of mute
	 *           name of the action
	 *           message if you're not on the channel
	 *           a color string.
	 *            ....
	 *           () ....
	 *           min access level.
	 *           mov cost.
	 */

	struct communication_type com_msgs[] =
		{
			{"   .\r\n",    // holler
			 "",
			 "   .",
			 KIYEL,
			 "",
			 "$g",
			 4,
			 25,
			 EPrf::kNoHoller},

			{"  .\r\n",    // shout
			 "",
			 "   .\r\n",
			 KIYEL,
			 "",
			 "$g",
			 2,
			 10,
			 EPrf::kNoShout},

			{"  .\r\n",    // gossip
			 "",
			 "   .\r\n",
			 KYEL,
			 "",
			 "$g",
			 3,
			 15,
			 EPrf::kNoGossip},

			{"    .\r\n",    // auction
			 "",
			 "   .\r\n",
			 KIYEL,
			 " ",
			 "$g  ",
			 2,
			 0,
			 EPrf::kNoAuction},
		};

	// to keep pets, etc from being ordered to shout
//  if (!ch->desc)
	if (AFF_FLAGGED(ch, EAffect::kCharmed))
		return;

	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar(SIELENCE, ch);
		return;
	}

	if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}

	if (PLR_FLAGGED(ch, EPlrFlag::kMuted) && subcmd != SCMD_AUCTION) {
		SendMsgToChar(com_msgs[subcmd].muted_msg, ch);
		return;
	}
	//if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kSoundproof) || ROOM_FLAGGED(ch->in_room, ERoomFlag::kTribune))
	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kSoundproof)) {
		SendMsgToChar(SOUNDPROOF, ch);
		return;
	}

	if (GetRealLevel(ch) < com_msgs[subcmd].min_lev && !GetRealRemort(ch)) {
		sprintf(buf1,
				"     %d ,    %s.\r\n",
				com_msgs[subcmd].min_lev, com_msgs[subcmd].action);
		SendMsgToChar(buf1, ch);
		return;
	}

	// make sure the char is on the channel
	if (PRF_FLAGS(ch).get(com_msgs[subcmd].noflag)) {
		SendMsgToChar(com_msgs[subcmd].no_channel, ch);
		return;
	}

	// skip leading spaces
	skip_spaces(&argument);

	// make sure that there is something there to say!
	if (!*argument && subcmd != SCMD_AUCTION) {
		sprintf(buf1, "! ,   ,  %s???\r\n", com_msgs[subcmd].action);
		SendMsgToChar(buf1, ch);
		return;
	}

	// set up the color on code
	strcpy(color_on, com_msgs[subcmd].color);

	// - :coded by 

#define MAX_UPPERS_CHAR_PRC 30
#define MAX_UPPERS_SEQ_CHAR 3

	if ((subcmd != SCMD_AUCTION) && (!IS_IMMORTAL(ch)) && (!ch->IsNpc())) {
		const unsigned int bad_smb_procent = MAX_UPPERS_CHAR_PRC;
		int bad_simb_cnt = 0, bad_seq_cnt = 0;

		//   
		for (int k = 0; argument[k] != '\0'; k++) {
			if (a_isupper(argument[k])) {
				bad_simb_cnt++;
				bad_seq_cnt++;
			} else
				bad_seq_cnt = 0;

			if ((bad_seq_cnt > 1) &&
				(((bad_simb_cnt * 100 / strlen(argument)) > bad_smb_procent) ||
					(bad_seq_cnt > MAX_UPPERS_SEQ_CHAR)))
				argument[k] = a_lcc(argument[k]);
		}
		//     
		if (!str_cmp(ch->get_last_tell().c_str(), argument)) {
			SendMsgToChar("      ?!\r\n", ch);
			return;
		}
		ch->set_last_tell(argument);
	}

	//        ,     
	if (!check_moves(ch, com_msgs[subcmd].move_cost))
		return;

	char out_str[kMaxStringLength];

	// first, set up strings to be given to the communicator
	if (subcmd == SCMD_AUCTION) {
		*buf = '\0';
		auction_drive(ch, argument);
		return;
	} else {
		/*  
		if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kTribune))
		{
			SendMsgToChar(SOUNDPROOF, ch);
			return;
		}
		*/
		if (PRF_FLAGS(ch).get(EPrf::kNoRepeat))
			SendMsgToChar(OK, ch);
		else {
			if (COLOR_LEV(ch) >= C_CMP) {
				snprintf(buf1, kMaxStringLength, "%s %s : '%s'%s", color_on,
						 com_msgs[subcmd].you_action, argument, KNRM);
			} else {
				snprintf(buf1, kMaxStringLength, " %s : '%s'",
						 com_msgs[subcmd].you_action, argument);
			}
			act(buf1, false, ch, 0, 0, kToChar | kToSleep);

			if (!ch->IsNpc()) {
				snprintf(buf1 + strlen(buf1), kMaxStringLength, "\r\n");
				ch->remember_add(buf1, Remember::ALL);
			}
		}
		switch (subcmd) {
			case SCMD_SHOUT: ign_flag = EIgnore::kShout;
				break;
			case SCMD_GOSSIP:
				if (PLR_FLAGGED(ch, EPlrFlag::kSpamer))
					return;
				ign_flag = EIgnore::kGossip;
				break;
			case SCMD_HOLLER:
				if (PLR_FLAGGED(ch, EPlrFlag::kSpamer))
					return;
				ign_flag = EIgnore::kHoller;
				break;
			default: ign_flag = 0;
		}
		snprintf(out_str, kMaxStringLength, "$n %s : '%s'", com_msgs[subcmd].hi_action, argument);
		if (IS_FEMALE(ch)) {
			if (!ch->IsNpc() && (subcmd == SCMD_GOSSIP)) {
				snprintf(buf1, kMaxStringLength, "%s%s  :'%s'%s\r\n", color_on, GET_NAME(ch), argument, KNRM);
				ch->remember_add(buf1, Remember::GOSSIP);
			}
			if (!ch->IsNpc() && (subcmd == SCMD_HOLLER)) {
				snprintf(buf1, kMaxStringLength, "%s%s  :'%s'%s\r\n", color_on, GET_NAME(ch), argument, KNRM);
				ch->remember_add(buf1, Remember::GOSSIP);
			}
		} else {
			if (!ch->IsNpc() && (subcmd == SCMD_GOSSIP)) {
				snprintf(buf1, kMaxStringLength, "%s%s  :'%s'%s\r\n", color_on, GET_NAME(ch), argument, KNRM);
				ch->remember_add(buf1, Remember::GOSSIP);
			}
			if (!ch->IsNpc() && (subcmd == SCMD_HOLLER)) {
				snprintf(buf1, kMaxStringLength, "%s%s  :'%s'%s\r\n", color_on, GET_NAME(ch), argument, KNRM);
				ch->remember_add(buf1, Remember::GOSSIP);
			}
		}
	}
	// now send all the strings out
	for (i = descriptor_list; i; i = i->next) {
		if (STATE(i) == CON_PLAYING && i != ch->desc && i->character &&
			!PRF_FLAGS(i->character).get(com_msgs[subcmd].noflag) &&
			!PLR_FLAGGED(i->character, EPlrFlag::kWriting) &&
			!ROOM_FLAGGED(i->character->in_room, ERoomFlag::kSoundproof) && GET_POS(i->character) > EPosition::kSleep) {
			if (ignores(i->character.get(), ch, ign_flag)) {
				continue;
			}

			if (subcmd == SCMD_SHOUT
				&& ((world[ch->in_room]->zone_rn != world[i->character->in_room]->zone_rn)
					|| !AWAKE(i->character))) {
				continue;
			}

			if (COLOR_LEV(i->character) >= C_NRM) {
				SendMsgToChar(color_on, i->character.get());
			}

			act(out_str, false, ch, 0, i->character.get(), kToVict | kToSleep | kToNotDeaf);
			if (COLOR_LEV(i->character) >= C_NRM) {
				SendMsgToChar(KNRM, i->character.get());
			}

			const std::string text = Remember::format_gossip(ch, i->character.get(), subcmd, argument);

			i->character->remember_add(text, Remember::ALL);
		}
	}
}

void do_mobshout(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	DescriptorData *i;

	// to keep pets, etc from being ordered to shout
	if (!(ch->IsNpc() || IS_IMMORTAL(ch)))
		return;
	if (AFF_FLAGGED(ch, EAffect::kCharmed))
		return;

	skip_spaces(&argument); //    
	sprintf(buf, "$n $g : '%s'", argument);

	// now send all the strings out
	for (i = descriptor_list; i; i = i->next) {
		if (STATE(i) == CON_PLAYING
			&& i->character
			&& !PLR_FLAGGED(i->character, EPlrFlag::kWriting)
			&& GET_POS(i->character) > EPosition::kSleep) {
			if (COLOR_LEV(i->character) >= C_NRM) {
				SendMsgToChar(KIYEL, i->character.get());
			}

			act(buf, false, ch, 0, i->character.get(), kToVict | kToSleep | kToNotDeaf);

			if (COLOR_LEV(i->character) >= C_NRM) {
				SendMsgToChar(KNRM, i->character.get());
			}
		}
	}
}

void do_pray_gods(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	char arg1[kMaxInputLength];
	DescriptorData *i;
	CharData *victim = nullptr;

	skip_spaces(&argument);

	if (!ch->IsNpc() && (PLR_FLAGGED(ch, EPlrFlag::kDumbed) || PLR_FLAGGED(ch, EPlrFlag::kMuted))) {
		SendMsgToChar("    , ,   ...\r\n", ch);
		return;
	}

	if (IS_IMMORTAL(ch)) {
		//     
		argument = one_argument(argument, arg1);
		skip_spaces(&argument);
		if (!*arg1) {
			SendMsgToChar("    ?\r\n", ch);
			return;
		}
		victim = get_player_vis(ch, arg1, EFind::kCharInWorld);
		if (victim == nullptr) {
			SendMsgToChar("   !\r\n", ch);
			return;
		}
	}

	if (!*argument) {
		sprintf(buf, "      ?\r\n");
		SendMsgToChar(buf, ch);
		return;
	}
	if (PRF_FLAGGED(ch, EPrf::kNoRepeat))
		SendMsgToChar(OK, ch);
	else {
		if (ch->IsNpc())
			return;
		if (IS_IMMORTAL(ch)) {
			sprintf(buf, "&R   %s : '%s'&n\r\n", GET_PAD(victim, 3), argument);
		} else {
			sprintf(buf, "&R      : '%s'&n\r\n", argument);
			SetWait(ch, 3, false);
		}
		SendMsgToChar(buf, ch);
		ch->remember_add(buf, Remember::PRAY_PERSONAL);
	}

	if (IS_IMMORTAL(ch)) {
		sprintf(buf, "&R%s %s  : '%s'&n\r\n", GET_NAME(ch), GET_CH_SUF_1(ch), argument);
		SendMsgToChar(buf, victim);
		victim->remember_add(buf, Remember::PRAY_PERSONAL);

		snprintf(buf1, kMaxStringLength, "&R%s %s %s : '%s&n\r\n",
				 GET_NAME(ch), GET_CH_SUF_1(ch), GET_PAD(victim, 2), argument);
		ch->remember_add(buf1, Remember::PRAY);

		snprintf(buf, kMaxStringLength, "&R%s %s   %s : '%s'&n\r\n",
				 GET_NAME(ch), GET_CH_SUF_1(ch), GET_PAD(victim, 1), argument);
	} else {
		snprintf(buf1, kMaxStringLength, "&R%s %s   : '%s&n\r\n",
				 GET_NAME(ch), GET_CH_SUF_1(ch), argument);
		ch->remember_add(buf1, Remember::PRAY);

		snprintf(buf, kMaxStringLength, "&R%s %s     : '%s'&n\r\n",
				 GET_NAME(ch), GET_CH_SUF_1(ch), argument);
	}

	for (i = descriptor_list; i; i = i->next) {
		if (STATE(i) == CON_PLAYING) {
			if ((IS_IMMORTAL(i->character.get())
				|| (GET_GOD_FLAG(i->character.get(), EGf::kDemigod)
					&& (GetRealLevel(ch) < 6)))
				&& (i->character.get() != ch)) {
				SendMsgToChar(buf, i->character.get());
				i->character->remember_add(buf, Remember::ALL);
			}
		}
	}
}

/**
*  .   ,    , /  .
*/
void do_offtop(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	if (ch->IsNpc() || GetRealLevel(ch) >= kLvlImmortal || PRF_FLAGGED(ch, EPrf::kStopOfftop)) {
		SendMsgToChar("?\r\n", ch);
		return;
	}

	if (PLR_FLAGGED(ch, EPlrFlag::kDumbed) || PLR_FLAGGED(ch, EPlrFlag::kMuted)) {
		SendMsgToChar("     !\r\n", ch);
		return;
	}
	//if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kSoundproof) || ROOM_FLAGGED(ch->in_room, ERoomFlag::kTribune))
	if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kSoundproof)) {
		SendMsgToChar(SOUNDPROOF, ch);
		return;
	}
	if (GetRealLevel(ch) < antispam::kMinOfftopLvl && !GetRealRemort(ch)) {
		SendMsgToChar(ch, "     %d ,    .\r\n",
					  antispam::kMinOfftopLvl);
		return;
	}
	if (!PRF_FLAGGED(ch, EPrf::kOfftopMode)) {
		SendMsgToChar("   .\r\n", ch);
		return;
	}
	skip_spaces(&argument);
	if (!*argument) {
		SendMsgToChar(ch, "  ,    .");
		return;
	}
	utils::ConvertToLow(argument);
	if (!strcmp(ch->get_last_tell().c_str(), argument)) {
		SendMsgToChar("      !?!\r\n", ch);
		return;
	}
	//             
	if (!antispam::check(ch, antispam::kOfftopMode)) {
		return;
	}
	ch->set_last_tell(argument);
	if (PLR_FLAGGED(ch, EPlrFlag::kSpamer)) //   ,   :)
		return;
	snprintf(buf, kMaxStringLength, "[] %s : '%s'\r\n", GET_NAME(ch), argument);
	snprintf(buf1, kMaxStringLength, "&c%s&n", buf);
	for (DescriptorData *i = descriptor_list; i; i = i->next) {
		//           ...
		//   ,  ?   34-!    -  ...
		if (STATE(i) == CON_PLAYING
			&& i->character
//   , 
//			&& (GetRealLevel(i->character) < kLvlImmortal || IS_IMPL(i->character))
			&& PRF_FLAGGED(i->character, EPrf::kOfftopMode)
			&& !PRF_FLAGGED(i->character, EPrf::kStopOfftop)
			&& !ignores(i->character.get(), ch, EIgnore::kOfftop)) {
			SendMsgToChar(i->character.get(), "%s%s%s", KCYN, buf, KNRM);
			i->character->remember_add(buf1, Remember::ALL);
		}
	}
	ch->remember_add(buf1, Remember::OFFTOP);
}

// shapirus
void ignore_usage(CharData *ch) {
	SendMsgToChar(" :  <|> <|> <|>\r\n"
				 " :\r\n"
				 "       \r\n"
				 "      \r\n", ch);
}

int ign_find_id(char *name, long *id) {
	for (std::size_t i = 0; i < player_table.size(); i++) {
		if (!str_cmp(name, player_table[i].name())) {
			if (player_table[i].level >= kLvlImmortal) {
				return 0;
			}

			*id = player_table[i].id();
			return 1;
		}
	}
	return -1;
}

const char *ign_find_name(long id) {
	for (std::size_t i = 0; i < player_table.size(); i++) {
		if (id == player_table[i].id()) {
			return player_table[i].name();
		}
	}

	return "-";
}

char *text_ignore_modes(unsigned long mode, char *buf) {
	buf[0] = 0;

	if (IS_SET(mode, EIgnore::kTell))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kSay))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kWhisper))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kAsk))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kEmote))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kShout))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kGossip))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kHoller))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kGroup))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kClan))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kAlliance))
		strcat(buf, " ");
	if (IS_SET(mode, EIgnore::kOfftop))
		strcat(buf, " ");
	return buf;
}

void do_ignore(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	char arg1[kMaxInputLength];
	char arg2[kMaxInputLength];
	char arg3[kMaxInputLength];
	unsigned int mode = 0, list_empty = 1, all = 0, flag = 0;
	long vict_id;
	char buf[256], buf1[256], name[50];

	argument = one_argument(argument, arg1);
	argument = one_argument(argument, arg2);
	argument = one_argument(argument, arg3);

//   --  
	if (arg1[0] && (!arg2[0] || !arg3[0])) {
		ignore_usage(ch);
		return;
	}
//       
	if (!arg1[0] && !arg2[0] && !arg3[0]) {
		sprintf(buf, "%s   :%s\r\n", CCWHT(ch, C_NRM), CCNRM(ch, C_NRM));
		SendMsgToChar(buf, ch);
		for (const auto &ignore : ch->get_ignores()) {
			if (!ignore->id)
				continue;
			if (ignore->id == -1) {
				strcpy(name, "");
			} else {
				strcpy(name, ign_find_name(ignore->id));
				name[0] = UPPER(name[0]);
			}
			sprintf(buf, "  %s: ", name);
			SendMsgToChar(buf, ch);
			mode = ignore->mode;
			SendMsgToChar(text_ignore_modes(mode, buf), ch);
			SendMsgToChar("\r\n", ch);
			list_empty = 0;
		}

		if (list_empty) {
			SendMsgToChar("   .\r\n", ch);
		}
		return;
	}

	if (utils::IsAbbr(arg2, ""))
		all = 1;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kTell;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kSay;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kWhisper;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kAsk;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kEmote;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kShout;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kGossip;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kHoller;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kGroup;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kClan;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kAlliance;
	else if (utils::IsAbbr(arg2, ""))
		flag = EIgnore::kOfftop;
	else {
		ignore_usage(ch);
		return;
	}

//  ""  id -1
	if (utils::IsAbbr(arg1, "")) {
		vict_id = -1;
	} else {
		// ,       
		//    ,     id
		switch (ign_find_id(arg1, &vict_id)) {
			case 0: SendMsgToChar("  ,   .\r\n", ch);
				return;
			case -1: SendMsgToChar("  ,  .\r\n", ch);
				return;
		}
	}

//    
	ignore_data::shared_ptr ignore = nullptr;
	for (const auto &ignore_i : ch->get_ignores()) {
		if (ignore_i->id == vict_id) {
			ignore = ignore_i;
			break;
		}
	}

	if (utils::IsAbbr(arg3, "")) {
//      ,   
		if (!ignore) {
			const auto cur = std::make_shared<ignore_data>();
			cur->id = vict_id;
			cur->mode = 0;
			ch->add_ignore(cur);
			ignore = cur;
		}
		mode = ignore->mode;
		if (all) {
			SET_BIT(mode, EIgnore::kTell);
			SET_BIT(mode, EIgnore::kSay);
			SET_BIT(mode, EIgnore::kWhisper);
			SET_BIT(mode, EIgnore::kAsk);
			SET_BIT(mode, EIgnore::kEmote);
			SET_BIT(mode, EIgnore::kShout);
			SET_BIT(mode, EIgnore::kGossip);
			SET_BIT(mode, EIgnore::kHoller);
			SET_BIT(mode, EIgnore::kGroup);
			SET_BIT(mode, EIgnore::kClan);
			SET_BIT(mode, EIgnore::kAlliance);
			SET_BIT(mode, EIgnore::kOfftop);
		} else {
			SET_BIT(mode, flag);
		}
		ignore->mode = mode;
	} else if (utils::IsAbbr(arg3, "")) {
		if (!ignore || ignore->id != vict_id) {
			if (vict_id == -1) {
				SendMsgToChar("      .\r\n", ch);
			} else {
				strcpy(name, ign_find_name(vict_id));
				name[0] = UPPER(name[0]);
				sprintf(buf,
						"     "
						" %s%s%s.\r\n", CCWHT(ch, C_NRM), name, CCNRM(ch, C_NRM));
				SendMsgToChar(buf, ch);
			}
			return;
		}
		mode = ignore->mode;
		if (all) {
			mode = 0;
		} else {
			REMOVE_BIT(mode, flag);
		}
		ignore->mode = mode;
		if (!mode)
			ignore->id = 0;
	} else {
		ignore_usage(ch);
		return;
	}
	if (mode) {
		if (ignore->id == -1) {
			sprintf(buf, "    :%s.\r\n", text_ignore_modes(ignore->mode, buf1));
			SendMsgToChar(buf, ch);
		} else {
			strcpy(name, ign_find_name(ignore->id));
			name[0] = UPPER(name[0]);
			sprintf(buf, "  %s%s%s  :%s.\r\n",
					CCWHT(ch, C_NRM), name, CCNRM(ch, C_NRM), text_ignore_modes(ignore->mode, buf1));
			SendMsgToChar(buf, ch);
		}
	} else {
		if (vict_id == -1) {
			SendMsgToChar("     .\r\n", ch);
		} else {
			strcpy(name, ign_find_name(vict_id));
			name[0] = UPPER(name[0]);
			sprintf(buf, "     %s%s%s.\r\n",
					CCWHT(ch, C_NRM), name, CCNRM(ch, C_NRM));
			SendMsgToChar(buf, ch);
		}
	}
}
