#include "cast.h"

#include "entities/char_data.h"
#include "game_magic/magic_utils.h"
#include "game_classes/classes_spell_slots.h"
#include "game_magic/spells_info.h"
#include "color.h"
#include "structs/global_objects.h"

auto FindSubstituteSpellId(CharData *ch, ESpell spell_id) {
	static const std::set<ESpell> healing_spells{
		ESpell::kCureLight,
		ESpell::kCureSerious,
		ESpell::kCureCritic,
		ESpell::kHeal};

	auto subst_spell_id{ESpell::kUndefined};
	if (CanUseFeat(ch, EFeat::kSpellSubstitute) && healing_spells.contains(spell_id)) {
		for (const auto &test_spell : MUD::Class(ch->GetClass()).spells) {
			auto test_spell_id = test_spell.GetId();
			if (GET_SPELL_MEM(ch, test_spell_id) &&
				MUD::Class(ch->GetClass()).spells[test_spell_id].GetCircle() ==
					MUD::Class(ch->GetClass()).spells[spell_id].GetCircle()) {
				subst_spell_id = test_spell_id;
				break;
			}
		}
	}

	return subst_spell_id;
}


/*
 * do_cast is the entry point for PC-casted spells.  It parses the arguments,
 * determines the spell number and finds a target, throws the die to see if
 * the spell can be cast, checks for sufficient mana and subtracts it, and
 * passes control to CastSpell().
 */
void DoCast(CharData *ch, char *argument, int/* cmd*/, int /*subcmd*/) {
	if (ch->IsNpc() && AFF_FLAGGED(ch, EAffect::kCharmed))
		return;
	if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
		SendMsgToChar("     .\r\n", ch);
		return;
	}
	if (ch->HasCooldown(ESkill::kGlobalCooldown)) {
		SendMsgToChar("   .\r\n", ch);
		return;
	}
	if (!ch->affected.empty()) {
		for (const auto &aff : ch->affected) {
			if (aff->location == EApply::kPlague && number(1, 100) < 10) { //  10%  
				SendMsgToChar("  ,       .\r\n",
							 ch);
				return;
			}
			if (aff->location == EApply::kMadness && number(1, 100) < 20) { //  20%  
				SendMsgToChar("  ,  ,   .\r\n", ch);
				return;
			}
		}
	}
	if (ch->is_morphed()) {
		SendMsgToChar("       .\r\n", ch);
		return;
	}
	if (!*argument) {
		SendMsgToChar("   ?\r\n", ch);
		return;
	}
	auto spell_name = strtok(argument, "'*!");
	if (!str_cmp(spell_name, argument)) {
		SendMsgToChar("       : *  !\r\n", ch);
		return;
	}
	const auto spell_id = FixNameAndFindSpellId(spell_name);
	if (spell_id == ESpell::kUndefined) {
		SendMsgToChar("     ?\r\n", ch);
		return;
	}
	if (const auto spell = MUD::Class(ch->GetClass()).spells[spell_id];
		(!IS_SET(GET_SPELL_TYPE(ch, spell_id), ESpellType::kTemp | ESpellType::kKnow) ||
		GetRealRemort(ch) < spell.GetMinRemort()) &&
		(GetRealLevel(ch) < kLvlGreatGod) && !ch->IsNpc()) {
		if (GetRealLevel(ch) < CalcMinSpellLvl(ch, spell_id)
			|| classes::CalcCircleSlotsAmount(ch, spell.GetCircle()) <= 0) {
			SendMsgToChar("     !\r\n", ch);
			return;
		} else {
			SendMsgToChar("   ,  ,  ...\r\n", ch);
			return;
		}
	}

	auto substitute_spell_id{ESpell::kUndefined};
	if (!GET_SPELL_MEM(ch, spell_id) && !IS_IMMORTAL(ch)) {
		substitute_spell_id = FindSubstituteSpellId(ch, spell_id);
		if (substitute_spell_id == ESpell::kUndefined) {
			SendMsgToChar("   ,    ...\r\n", ch);
			return;
		}
	} else {
		substitute_spell_id = spell_id;
	}


	if (auto target_name = strtok(nullptr, "\0"); target_name != nullptr) {
		one_argument(target_name, arg);
	} else {
		*arg = '\0';
	}

	CharData *tch;
	ObjData *tobj;
	RoomData *troom;
	auto target = FindCastTarget(spell_id, arg, ch, &tch, &tobj, &troom);
	if (target && (tch == ch) && MUD::Spell(spell_id).IsViolent()) {
		SendMsgToChar("      !\r\n", ch);
		return;
	}
	if (!target) {
		SendMsgToChar("    !\r\n", ch);
		return;
	}
	if (!IS_SET(GET_SPELL_TYPE(ch, spell_id), ESpellType::kTemp) &&
		ROOM_FLAGGED(ch->in_room, ERoomFlag::kDominationArena)) {
		SendMsgToChar("        !\r\n", ch);
		return;
	}

	ch->SetCast(ESpell::kUndefined, ESpell::kUndefined, nullptr, nullptr, nullptr);
	if (!CalcCastSuccess(ch, tch, ESaving::kStability, spell_id)) {
		if (!(IS_IMMORTAL(ch) || GET_GOD_FLAG(ch, EGf::kGodsLike)))
			SetWaitState(ch, kBattleRound);
		if (GET_SPELL_MEM(ch, substitute_spell_id)) {
			GET_SPELL_MEM(ch, substitute_spell_id)--;
		}
		if (!ch->IsNpc() && !IS_IMMORTAL(ch) && PRF_FLAGGED(ch, EPrf::kAutomem)) {
			MemQ_remember(ch, substitute_spell_id);
		}
		affect_total(ch);
		if (!tch || !SendSkillMessages(0, ch, tch, spell_id)) {
			SendMsgToChar("   !\r\n", ch);
		}
	} else {
		if (ch->GetEnemy() && !IS_IMPL(ch)) {
			ch->SetCast(spell_id, substitute_spell_id, tch, tobj, troom);
			sprintf(buf, "    %s'%s'%s%s.\r\n",
					CCCYN(ch, C_NRM), MUD::Spell(spell_id).GetCName(), CCNRM(ch, C_NRM),
					tch == ch ? "  " : tch ? "  $N3" : tobj ? "  $o3" : troom ? "  " : "");
			act(buf, false, ch, tobj, tch, kToChar);
		} else if (CastSpell(ch, tch, tobj, troom, spell_id, substitute_spell_id) >= 0) {
			if (!(IS_IMMORTAL(ch) || ch->get_wait() > 0))
				SetWaitState(ch, kBattleRound);
		}
	}
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
