/**************************************************************************
*   File: exchange.cpp                                 Part of Bylins     *
*  Usage: Exchange functions used by the MUD                              *
*                                                                         *
*                                                                         *
*  $Author$                                                         *
*  $Date$                                           *
*  $Revision$                                                      *
************************************************************************ */

#include "exchange.h"

#include "obj_prototypes.h"
#include "entities/world_characters.h"
#include "entities/obj_data.h"
#include "entities/entities_constants.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "color.h"
#include "game_crafts/im.h"
#include "constants.h"
#include "game_skills/skills.h"
#include "entities/char_data.h"
#include "entities/char_player.h"
#include "game_mechanics/named_stuff.h"
#include "modify.h"
#include "entities/room_data.h"
#include "communication/mail.h"
#include "obj_save.h"
#include "game_fight/pk.h"
#include "utils/logger.h"
#include "utils/objects_filter.h"
#include "utils/utils.h"
#include "structs/structs.h"
#include "sysdep.h"
#include "conf.h"

#include <stdexcept>
#include <sstream>

//  -.
extern int get_buf_line(char **source, char *target);
extern int get_buf_lines(char **source, char *target);

extern int invalid_anti_class(CharData *ch, const ObjData *obj);
extern int invalid_unique(CharData *ch, const ObjData *obj);
extern int invalid_no_class(CharData *ch, const ObjData *obj);
extern int invalid_align(CharData *ch, const ObjData *obj);
extern char *diag_weapon_to_char(const CObjectPrototype *obj, int show_wear);
extern const char *diag_obj_timer(const ObjData *obj);
extern char *diag_timer_to_char(const ObjData *obj);
extern void mort_show_obj_values(const ObjData *obj, CharData *ch, int fullness, bool enhansed_scroll);
extern void SetWait(CharData *ch, int waittime, int victim_in_room);
extern bool is_dig_stone(ObjData *obj);

// -
int exchange_exhibit(CharData *ch, char *arg);
int exchange_change_cost(CharData *ch, char *arg);
int exchange_withdraw(CharData *ch, char *arg);
int exchange_information(CharData *ch, char *arg);
int exchange_identify(CharData *ch, char *arg);
int exchange_purchase(CharData *ch, char *arg);
int exchange_offers(CharData *ch, char *arg);
int exchange_setfilter(CharData *ch, char *arg);

int exchange_database_load();
int exchange_database_reload(bool loadbackup);
void check_exchange(ObjData *obj);
void extract_exchange_item(ExchangeItem *item);
int get_unique_lot();
void message_exchange(char *message, CharData *ch, ExchangeItem *j);
void show_lots(char *filter, short int show_type, CharData *ch);
int parse_exch_filter(ParseFilter &filter, char *buf, bool parse_affects);
void clear_exchange_lot(ExchangeItem *lot);
extern void obj_info(CharData *ch, ObjData *obj, char buf[kMaxStringLength]);

void do_exchange(CharData *ch, char *argument, int cmd, int subcmd);

ExchangeItem *create_exchange_item();

ExchangeItem *exchange_item_list = nullptr;
std::vector<bool> lot_usage;

char arg1[kMaxInputLength], arg2[kMaxInputLength];

char info_message[] = ("  <> <>        -    \r\n"
					   "  <#> <>                -     \r\n"
					   "  <#>                      -     \r\n"
					   "  <#>                 -   \r\n"
					   "  <#>             -   (  110 )\r\n"
					   "  <#>                     -  \r\n"
					   "   <>         -  \r\n"
					   "               - \r\n"
					   "   <>         -  \r\n"
					   "   <>        -  \r\n"
					   "   <>       -    \r\n"
					   //					   "   <>      -   \r\n"
					   //					   "   <>     -   \r\n"
					   //					   "   <>     -   \r\n"
					   "   <>      -  \r\n"
					   "   <>       -  \r\n"
					   "   <> -  \r\n"
					   "   <>      -  \r\n"
					   "   .    -    (  55 )\r\n"
					   "  <>               -    \r\n");

int exchange(CharData *ch, void * /*me*/, int cmd, char *argument) {
	if (CMD_IS("exchange") || CMD_IS("")) {
		if (ch->IsNpc())
			return 0;
		if (AFF_FLAGGED(ch, EAffect::kSilence) || AFF_FLAGGED(ch, EAffect::kStrangled)) {
			SendMsgToChar(" ,    .\r\n", ch);
			return 1;
		}
		if (!ch->IsNpc() && PLR_FLAGGED(ch, EPlrFlag::kDumbed)) {
			SendMsgToChar("    !\r\n", ch);
			return 1;
		}
		/*
				if (EPlrFlag::FLAGGED(ch, EPlrFlag::MUTE)) {
					SendMsgToChar("    .\r\n", ch);
					return 1;
				}
		*/
		if (GetRealLevel(ch) < EXCHANGE_MIN_CHAR_LEV && !GetRealRemort(ch)) {
			sprintf(buf1,
					"     %d ,   .\r\n",
					EXCHANGE_MIN_CHAR_LEV);
			SendMsgToChar(buf1, ch);
			return 1;
		}
		if (NORENTABLE(ch)) {
			SendMsgToChar("   .\r\n", ch);
			return 1;
		}

		argument = one_argument(argument, arg1);

		if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "exhibit"))
			exchange_exhibit(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "cost"))
			exchange_change_cost(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "withdraw"))
			exchange_withdraw(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "information"))
			exchange_information(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "identify"))
			exchange_identify(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "purchase"))
			exchange_purchase(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "offers"))
			exchange_offers(ch, argument);
		else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "filter"))
			exchange_setfilter(ch, argument);
		else if (utils::IsAbbr(arg1, "save") && (GetRealLevel(ch) >= kLvlImplementator))
			exchange_database_save();
		else if (utils::IsAbbr(arg1, "savebackup") && (GetRealLevel(ch) >= kLvlImplementator))
			exchange_database_save(true);
		else if (utils::IsAbbr(arg1, "reload") && (GetRealLevel(ch) >= kLvlImplementator))
			exchange_database_reload(false);
		else if (utils::IsAbbr(arg1, "reloadbackup") && (GetRealLevel(ch) >= kLvlImplementator))
			exchange_database_reload(true);
		else
			SendMsgToChar(info_message, ch);

		return 1;
	} else
		return 0;

}

int exchange_exhibit(CharData *ch, char *arg) {

	char obj_name[kMaxInputLength];
	int item_cost = 0, lot;
	ObjData *obj;
	char tmpbuf[kMaxInputLength];
	ExchangeItem *item = nullptr;
	ExchangeItem *j, *next_thing = nullptr;
	int counter;
	int counter_ming; // 
	int tax;    //

	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (GetRealLevel(ch) >= kLvlImmortal && GetRealLevel(ch) < kLvlImplementator) {
		SendMsgToChar(",     ,     .\r\n", ch);
		return false;
	}

	arg = one_argument(arg, obj_name);
	arg = one_argument(arg, arg2);
	if (!*obj_name) {
		SendMsgToChar(":     \r\n", ch);
		return false;
	}
	if (!sscanf(arg2, "%d", &item_cost))
		item_cost = 0;
	if (!*obj_name) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}
	if (!(obj = get_obj_in_list_vis(ch, obj_name, ch->carrying))) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	if (!bloody::handle_transfer(ch, nullptr, obj))
		return false;
	if (item_cost > 400000 && obj->has_flag(EObjFlag::KSetItem)) {
		SendMsgToChar("      .\r\n", ch);
		return false;
	}
	if (GET_OBJ_TYPE(obj) != EObjType::kBook) {
		if (obj->has_flag(EObjFlag::kNorent)
			|| obj->has_flag(EObjFlag::kNosell)
			|| obj->has_flag(EObjFlag::kZonedacay)
			|| obj->has_flag(EObjFlag::kRepopDecay)
			|| GET_OBJ_RNUM(obj) < 0) {
			SendMsgToChar("     .\r\n", ch);
			return false;
		}
	}
	if (obj->has_flag(EObjFlag::kDecay) || obj->has_flag(EObjFlag::kNodrop)
		|| GET_OBJ_COST(obj) <= 0
		|| obj->get_owner() > 0) {
		SendMsgToChar("     .\r\n", ch);
		return false;
	}
	if (obj->get_contains()) {
		sprintf(tmpbuf, " %s  .\r\n", obj->get_PName(3).c_str());
		SendMsgToChar(tmpbuf, ch);
		return false;
	} else if (SetSystem::is_big_set(obj, true)) {
		snprintf(buf, kMaxStringLength, "%s     .\r\n",
				 obj->get_PName(0).c_str());
		SendMsgToChar(CAP(buf), ch);
		return false;
	}

	if (item_cost <= 0) {
		item_cost = std::max(1, GET_OBJ_COST(obj));
	}

	tax = (GET_OBJ_TYPE(obj) != EObjType::kMagicIngredient)
		  ? EXCHANGE_EXHIBIT_PAY + (int) (item_cost * EXCHANGE_EXHIBIT_PAY_COEFF)
		  : (int) (item_cost * EXCHANGE_EXHIBIT_PAY_COEFF / 2);
	if ((ch->get_total_gold() < tax)
		&& (GetRealLevel(ch) < kLvlImplementator)) {
		SendMsgToChar("      !\r\n", ch);
		return false;
	}
	for (j = exchange_item_list, counter = 0, counter_ming = 0;
		 j && (counter + (counter_ming / 20) <= EXCHANGE_MAX_EXHIBIT_PER_CHAR + (GetRealRemort(ch) * 2));
		 j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_SELLERID(j) == GET_IDNUM(ch)) {
			if (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kMagicIngredient
				&& GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kIngredient) {
				counter++;
			} else {
				counter_ming++;
			}
		}
	}

	if (counter + (counter_ming / 20) >= EXCHANGE_MAX_EXHIBIT_PER_CHAR + (GetRealRemort(ch) * 2)) {
		SendMsgToChar("       !\r\n", ch);
		return false;
	}

	if ((lot = get_unique_lot()) <= 0) {
		SendMsgToChar(" !\r\n", ch);
		return false;
	}
	item = create_exchange_item();
	GET_EXCHANGE_ITEM_LOT(item) = lot;
	GET_EXCHANGE_ITEM_SELLERID(item) = GET_IDNUM(ch);
	GET_EXCHANGE_ITEM_COST(item) = item_cost;
	item->time = time(0);
	skip_spaces(&arg);
	if (*arg)
		GET_EXCHANGE_ITEM_COMMENT(item) = str_dup(arg);
	else
		GET_EXCHANGE_ITEM_COMMENT(item) = nullptr;

	if (check_unlimited_timer(obj)) //    1 
		obj->set_timer(10080);
	GET_EXCHANGE_ITEM(item) = obj;
	RemoveObjFromChar(obj);

	sprintf(tmpbuf, "    $O3 ( %d)  %d %s.",
			GET_EXCHANGE_ITEM_LOT(item), item_cost, GetDeclensionInNumber(item_cost, EWhat::kMoneyU));
	act(tmpbuf, false, ch, 0, obj, kToChar);
	sprintf(tmpbuf, " :   (%d) - %s -  %d %s. \r\n",
			GET_EXCHANGE_ITEM_LOT(item), obj->get_PName(0).c_str(), item_cost,
			GetDeclensionInNumber(item_cost, EWhat::kMoneyA));
	message_exchange(tmpbuf, ch, item);

	ch->remove_both_gold(tax);

	if (EXCHANGE_SAVEONEVERYOPERATION) {
		exchange_database_save();
		Crash_crashsave(ch);
		ch->save_char();
	}
	return (true);
}

int exchange_change_cost(CharData *ch, char *arg) {
	ExchangeItem *item = nullptr, *j, *next_thing = nullptr;
	int lot, newcost, pay;
	char tmpbuf[kMaxInputLength];

	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (GetRealLevel(ch) >= kLvlImmortal && GetRealLevel(ch) < kLvlImplementator) {
		SendMsgToChar(",     ,     .\r\n", ch);
		return false;
	}
	if (sscanf(arg, "%d %d", &lot, &newcost) != 2) {
		SendMsgToChar(" :   <> < >.\r\n", ch);
		return false;
	}
	for (j = exchange_item_list; j && (!item); j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_LOT(j) == lot)
			item = j;
	}
	if ((lot < 0) || (!item)) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}
	if ((GET_EXCHANGE_ITEM_SELLERID(item) != GET_IDNUM(ch)) && (GetRealLevel(ch) < kLvlImplementator)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	if (newcost == GET_EXCHANGE_ITEM_COST(item)) {
		SendMsgToChar("     .\r\n", ch);
		return false;
	}
	if (newcost <= 0) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	pay = newcost - GET_EXCHANGE_ITEM_COST(item);
	if (pay > 0)
		if ((ch->get_total_gold() < (pay * EXCHANGE_EXHIBIT_PAY_COEFF)) && (GetRealLevel(ch) < kLvlImplementator)) {
			SendMsgToChar("      !\r\n", ch);
			return false;
		}

	GET_EXCHANGE_ITEM_COST(item) = newcost;
	if (pay > 0) {
		ch->remove_both_gold(static_cast<long>(pay * EXCHANGE_EXHIBIT_PAY_COEFF));
	}

	sprintf(tmpbuf,
			"   %d %s  %s ( %d).\r\n",
			newcost,
			GetDeclensionInNumber(newcost, EWhat::kMoneyU),
			GET_EXCHANGE_ITEM(item)->get_PName(3).c_str(),
			GET_EXCHANGE_ITEM_LOT(item));
	SendMsgToChar(tmpbuf, ch);
	sprintf(tmpbuf,
			" :  (%d) - %s -     %d %s.\r\n",
			GET_EXCHANGE_ITEM_LOT(item),
			GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(),
			newcost,
			GetDeclensionInNumber(newcost, EWhat::kMoneyA));
	message_exchange(tmpbuf, ch, item);
	SetWait(ch, 2, false);

	return true;
}

int exchange_withdraw(CharData *ch, char *arg) {
	ExchangeItem *item = nullptr, *j, *next_thing = nullptr;
	int lot;
	char tmpbuf[kMaxInputLength];

	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (GetRealLevel(ch) >= kLvlImmortal && GetRealLevel(ch) < kLvlImplementator) {
		SendMsgToChar(",     ,     .\r\n", ch);
		return false;
	}

	if (!sscanf(arg, "%d", &lot)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	for (j = exchange_item_list; j && (!item); j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_LOT(j) == lot)
			item = j;
	}
	if ((lot < 0) || (!item)) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}
	if ((GET_EXCHANGE_ITEM_SELLERID(item) != GET_IDNUM(ch)) && (GetRealLevel(ch) < kLvlImplementator)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	act("  $O3  .\r\n", false, ch, 0, GET_EXCHANGE_ITEM(item), kToChar);
	if (GET_EXCHANGE_ITEM_SELLERID(item) != GET_IDNUM(ch)) {
		sprintf(tmpbuf, " :  %d(%s) %s   .\r\n", lot,
				GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)));
	} else {
		sprintf(tmpbuf, " :  %d(%s) %s   .\r\n", lot,
				GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)));
	}
	message_exchange(tmpbuf, ch, item);
	if (check_unlimited_timer(GET_EXCHANGE_ITEM(item))) //      
		GET_EXCHANGE_ITEM(item)->set_timer(obj_proto.at(GET_OBJ_RNUM(GET_EXCHANGE_ITEM(item)))->get_timer());
	PlaceObjToInventory(GET_EXCHANGE_ITEM(item), ch);
	clear_exchange_lot(item);

	if (EXCHANGE_SAVEONEVERYOPERATION) {
		exchange_database_save();
		Crash_crashsave(ch);
		ch->save_char();
	}

	return true;
}

int exchange_information(CharData *ch, char *arg) {
	ExchangeItem *item = nullptr, *j, *next_thing = nullptr;
	int lot;
	char buf[kMaxStringLength], buf2[kMaxInputLength];

	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (!sscanf(arg, "%d", &lot)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	for (j = exchange_item_list; j && (!item); j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_LOT(j) == lot)
			item = j;
	}
	if ((lot < 0) || (!item)) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}

	sprintf(buf, " %d.  %d\r\n", GET_EXCHANGE_ITEM_LOT(item), GET_EXCHANGE_ITEM_COST(item));

	sprintf(buf + strlen(buf), " \"%s\", ", GET_EXCHANGE_ITEM(item)->get_short_description().c_str());
	if (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(item)) == EObjType::kWand
		|| GET_OBJ_TYPE(GET_EXCHANGE_ITEM(item)) == EObjType::kStaff) {
		if (GET_OBJ_VAL(GET_EXCHANGE_ITEM(item), 2) < GET_OBJ_VAL(GET_EXCHANGE_ITEM(item), 1)) {
			strcat(buf, "(/), ");
		}
	}
	strcat(buf, "  ");
	sprinttype(GET_OBJ_TYPE(GET_EXCHANGE_ITEM(item)), item_types, buf2);
	if (*buf2) {
		strcat(buf, buf2);
		strcat(buf, "\n");
	};
	strcat(buf, diag_weapon_to_char(GET_EXCHANGE_ITEM(item), true));
	strcat(buf, diag_timer_to_char(GET_EXCHANGE_ITEM(item)));
	strcat(buf, "\r\n");
	obj_info(ch, GET_EXCHANGE_ITEM(item), buf);
	strcat(buf, "\n");
	if (invalid_anti_class(ch, GET_EXCHANGE_ITEM(item)) || invalid_unique(ch, GET_EXCHANGE_ITEM(item))
		|| NamedStuff::check_named(ch, GET_EXCHANGE_ITEM(item), 0)) {
		sprintf(buf2, "   !");
		strcat(buf, buf2);
		strcat(buf, "\n");
	}
	if (HaveIncompatibleAlign(ch, GET_EXCHANGE_ITEM(item)) || invalid_no_class(ch, GET_EXCHANGE_ITEM(item))) {
		sprintf(buf2, "     .");
		strcat(buf, buf2);
		strcat(buf, "\n");
	}
	sprintf(buf2, "%s",
			get_name_by_id(GET_EXCHANGE_ITEM_SELLERID(item)) ?
			get_name_by_id(GET_EXCHANGE_ITEM_SELLERID(item)) : "(null)");
	*buf2 = UPPER(*buf2);
	strcat(buf, " ");
	strcat(buf, buf2);
	strcat(buf, "\n");

	if (GET_EXCHANGE_ITEM_COMMENT(item)) {
		strcat(buf, "    : ");
		sprintf(buf2, "'%s'.", GET_EXCHANGE_ITEM_COMMENT(item));
		strcat(buf, buf2);
		strcat(buf, "\n");
	}
	SendMsgToChar(buf, ch);
	return true;
}

int exchange_identify(CharData *ch, char *arg) {
	ExchangeItem *item = nullptr, *j, *next_thing = nullptr;
	int lot;
	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (!sscanf(arg, "%d", &lot)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	for (j = exchange_item_list; j && (!item); j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_LOT(j) == lot)
			item = j;
	}
	if ((lot < 0) || (!item)) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}

	if (GetRealLevel(ch) >= kLvlImmortal && GetRealLevel(ch) < kLvlImplementator) {
		SendMsgToChar(",       .\r\n", ch);
		return false;
	}
	if ((ch->get_total_gold() < (EXCHANGE_IDENT_PAY)) && (GetRealLevel(ch) < kLvlImplementator)) {
		SendMsgToChar("      !\r\n", ch);
		return false;
	}
	bool full = false;
	mort_show_obj_values(GET_EXCHANGE_ITEM(item), ch, 200, full);    //200 -  
	ch->remove_both_gold(EXCHANGE_IDENT_PAY);
	SendMsgToChar(ch, "\r\n%s         %d %s%s\r\n",
				  CCIGRN(ch, C_NRM), EXCHANGE_IDENT_PAY,
				  GetDeclensionInNumber(EXCHANGE_IDENT_PAY, EWhat::kMoneyU), CCNRM(ch, C_NRM));

	return true;
}

CharData *get_char_by_id(int id) {
	for (const auto &i : character_list) {
		if (!i->IsNpc() && GET_IDNUM(i) == id) {
			return i.get();
		}
	}
	return 0;
}

int exchange_purchase(CharData *ch, char *arg) {
	ExchangeItem *item = nullptr, *j, *next_thing = nullptr;
	int lot;
	char tmpbuf[kMaxInputLength];
	CharData *seller;

	if (!*arg) {
		SendMsgToChar(info_message, ch);
		return false;
	}
	if (GetRealLevel(ch) >= kLvlImmortal && GetRealLevel(ch) < kLvlImplementator) {
		SendMsgToChar(",     ,     .\r\n", ch);
		return false;
	}

	if (!sscanf(arg, "%d", &lot)) {
		SendMsgToChar("   .\r\n", ch);
		return false;
	}
	for (j = exchange_item_list; j && (!item); j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM_LOT(j) == lot)
			item = j;
	}
	if ((lot < 0) || (!item)) {
		SendMsgToChar("  .\r\n", ch);
		return false;
	}
	if (GET_EXCHANGE_ITEM_SELLERID(item) == GET_IDNUM(ch)) {
		SendMsgToChar("   .   '  <>'\r\n", ch);
		return false;
	}
	if ((ch->get_total_gold() < (GET_EXCHANGE_ITEM_COST(item))) && (GetRealLevel(ch) < kLvlImplementator)) {
		SendMsgToChar("         !\r\n", ch);
		return false;
	}

	seller = get_char_by_id(GET_EXCHANGE_ITEM_SELLERID(item));

	if (seller == nullptr) {
		const char *seller_name = get_name_by_id(GET_EXCHANGE_ITEM_SELLERID(item));

		auto seller_ptr = std::make_unique<Player>();
		seller = seller_ptr.get(); // TODO:   
		if (seller_name == nullptr
			|| load_char(seller_name, seller) < 0) {
			ch->remove_both_gold(GET_EXCHANGE_ITEM_COST(item));

			//edited by WorM 2011.05.21
			act("  $O3  .\r\n", false, ch, 0, GET_EXCHANGE_ITEM(item), kToChar);
			sprintf(tmpbuf, " :  %d(%s) %s  %d %s.\r\n", lot,
					GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)),
					GET_EXCHANGE_ITEM_COST(item), GetDeclensionInNumber(GET_EXCHANGE_ITEM_COST(item), EWhat::kMoneyU));
			//end by WorM

			message_exchange(tmpbuf, ch, item);
			if (check_unlimited_timer(GET_EXCHANGE_ITEM(item))) //      
				GET_EXCHANGE_ITEM(item)->set_timer(obj_proto.at(GET_OBJ_RNUM(GET_EXCHANGE_ITEM(item)))->get_timer());
			PlaceObjToInventory(GET_EXCHANGE_ITEM(item), ch);
			clear_exchange_lot(item);
			if (EXCHANGE_SAVEONEVERYOPERATION) {
				exchange_database_save();
				Crash_crashsave(ch);
				ch->save_char();
			}

			return true;
		}

		seller->add_bank(GET_EXCHANGE_ITEM_COST(item), true);
		ch->remove_both_gold(GET_EXCHANGE_ITEM_COST(item), true);
//Polud   ,     
		if (NOTIFY_EXCH_PRICE(seller) && GET_EXCHANGE_ITEM_COST(item) >= NOTIFY_EXCH_PRICE(seller)) {
			sprintf(tmpbuf, " :  %d(%s) %s. %d %s    .\r\n", lot,
					GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)),
					GET_EXCHANGE_ITEM_COST(item), GetDeclensionInNumber(GET_EXCHANGE_ITEM_COST(item), EWhat::kMoneyA));
			mail::add_by_id(GET_EXCHANGE_ITEM_SELLERID(item), -1, tmpbuf);
		}
//-Polud
		seller->save_char();

		act("  $O3  .\r\n", false, ch, 0, GET_EXCHANGE_ITEM(item), kToChar);
		sprintf(tmpbuf, " :  %d(%s) %s  %d %s.\r\n", lot,
				GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)),
				GET_EXCHANGE_ITEM_COST(item), GetDeclensionInNumber(GET_EXCHANGE_ITEM_COST(item), EWhat::kMoneyU));
		message_exchange(tmpbuf, ch, item);
		if (check_unlimited_timer(GET_EXCHANGE_ITEM(item))) //      
			GET_EXCHANGE_ITEM(item)->set_timer(obj_proto.at(GET_OBJ_RNUM(GET_EXCHANGE_ITEM(item)))->get_timer());
		PlaceObjToInventory(GET_EXCHANGE_ITEM(item), ch);
		clear_exchange_lot(item);
		if (EXCHANGE_SAVEONEVERYOPERATION) {
			exchange_database_save();
			Crash_crashsave(ch);
			ch->save_char();
		}

		return true;
	} else {
		seller->add_bank(GET_EXCHANGE_ITEM_COST(item), true);
		ch->remove_both_gold(GET_EXCHANGE_ITEM_COST(item), true);

		act("  $O3  .\r\n", false, ch, 0, GET_EXCHANGE_ITEM(item), kToChar);
		sprintf(tmpbuf, " :  %d(%s) %s  %d %s.\r\n", lot,
				GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)),
				GET_EXCHANGE_ITEM_COST(item), GetDeclensionInNumber(GET_EXCHANGE_ITEM_COST(item), EWhat::kMoneyU));
		message_exchange(tmpbuf, seller, item);
		sprintf(tmpbuf, " :  %d(%s) %s. %d %s    .\r\n", lot,
				GET_EXCHANGE_ITEM(item)->get_PName(0).c_str(), GET_OBJ_SUF_6(GET_EXCHANGE_ITEM(item)),
				GET_EXCHANGE_ITEM_COST(item), GetDeclensionInNumber(GET_EXCHANGE_ITEM_COST(item), EWhat::kMoneyA));
		act(tmpbuf, false, seller, 0, nullptr, kToChar);

		PlaceObjToInventory(GET_EXCHANGE_ITEM(item), ch);
		clear_exchange_lot(item);
		if (EXCHANGE_SAVEONEVERYOPERATION) {
			exchange_database_save();
			Crash_crashsave(ch);
			ch->save_char();
			Crash_crashsave(seller);
			seller->save_char();
		}

		return true;
	}
}

/**
*   .
* \return false - 
*         true - 
*/
bool correct_filter_length(CharData *ch, const char *str) {
	if (strlen(str) >= FILTER_LENGTH) {
		SendMsgToChar(ch, "  ,  : %d .\r\n", FILTER_LENGTH - 1);
		return false;
	}
	return true;
}

int exchange_offers(CharData *ch, char *arg) {
//  .    ,  .
	char filter[kMaxInputLength];
	char multifilter[kMaxStringLength];
	short int show_type;
	bool ignore_filter;
	char arg3[kMaxInputLength], arg4[kMaxInputLength];

	/*
	show_type
	0 - 
	1 - 
	2 - 
	3 - 
	4 - 
	5 - 
	6 - 
	7 - 
	8 - 
	*/

	memset(filter, 0, FILTER_LENGTH);
	memset(multifilter, 0, FILTER_LENGTH);

	(arg = one_argument(arg, arg1));
	(arg = one_argument(arg, arg2));
	(arg = one_argument(arg, arg3));
	(arg = one_argument(arg, arg4));

	ignore_filter = ((*arg2 == '*') || (*arg3 == '*') || (*arg4 == '*'));

	if (*arg2 == '!') {
		strcpy(multifilter, arg2 + 1);
	}
	if (*arg3 == '!') {
		strcpy(multifilter, arg3 + 1);
	}
	if (*arg4 == '!') {
		strcpy(multifilter, arg4 + 1);
	}

	if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "all")) {
		show_type = 0;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!')) {
			snprintf(filter, FILTER_LENGTH, "%s", arg2);
		}
		if (*multifilter) {
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "mine")) {
		show_type = 1;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
		if (*multifilter) {
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "runes")) {
		show_type = 2;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
		if (*multifilter) {
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "armor")) {
		show_type = 3;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
		if (*multifilter) {
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "weapons")) {
		show_type = 4;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
		if (*multifilter) {
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "books")) {
		show_type = 5;
		if ((*arg2) && (*arg2 != '*')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "ingradients")) {
		show_type = 6;
		if ((*arg2) && (*arg2 != '*')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "other")) {
		show_type = 7;
		if ((*arg2) && (*arg2 != '*')) {
			sprintf(buf, "%s", filter);
			snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
		}
	} else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "last")) {
		show_type = 8;
		//  3     
	}
/*	else if (utils::is_abbrev(arg1, "") || utils::is_abbrev(arg1, ""))
	{
		show_type = 9;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!'))
		{
			sprintf(filter, "%s %s", filter, arg2);
		}
		if (*multifilter)
		{
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	}
	else if (utils::is_abbrev(arg1, "") || utils::is_abbrev(arg1, ""))
	{
		show_type = 10;
		if ((*arg2) && (*arg2 != '*') && (*arg2 != '!'))
		{
			sprintf(filter, "%s %s", filter, arg2);
		}
		if (*multifilter)
		{
			strcat(filter, " ");
			strcat(filter, multifilter);
		}
	}
*/
	else if (utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "affect")) {
		if (ch->get_total_gold() < EXCHANGE_IDENT_PAY / 2 && GetRealLevel(ch) < kLvlImplementator) {
			SendMsgToChar("      !\r\n", ch);
			return 0;
		}
		if (*arg2 == '\0') {
			SendMsgToChar("  !\r\n", ch);
			return 0;
		}
		show_type = 11;
		sprintf(buf, "%s", filter);
		snprintf(filter, FILTER_LENGTH, "%s %s", buf, arg2);
	} else {
		SendMsgToChar(info_message, ch);
		return 0;
	}

	if (!ignore_filter && EXCHANGE_FILTER(ch))
		snprintf(multifilter, kMaxStringLength, "%s %s", EXCHANGE_FILTER(ch), filter);
	else
		strcpy(multifilter, filter);

	if (!correct_filter_length(ch, multifilter)) {
		return 0;
	}

	show_lots(multifilter, show_type, ch);
	return 1;
}

int exchange_setfilter(CharData *ch, char *arg) {
	if (!*arg) {
		if (!EXCHANGE_FILTER(ch)) {
			SendMsgToChar("   .\r\n", ch);
			return true;
		}
		ParseFilter params(ParseFilter::EXCHANGE);
		if (!parse_exch_filter(params, EXCHANGE_FILTER(ch), false)) {
			free(EXCHANGE_FILTER(ch));
			EXCHANGE_FILTER(ch) = nullptr;
			SendMsgToChar("   \r\n", ch);
		} else {
			SendMsgToChar(ch, "   : %s.\r\n",
						  EXCHANGE_FILTER(ch));
		}
		return true;
	}

	char tmpbuf[kMaxInputLength];
	char filter[kMaxInputLength];

	skip_spaces(&arg);
	strcpy(filter, arg);
	if (!correct_filter_length(ch, filter)) {
		return false;
	}
	if (!strncmp(filter, "", 3)) {
		if (EXCHANGE_FILTER(ch)) {
			snprintf(tmpbuf, sizeof(tmpbuf),
					 "  : %s.   .\r\n",
					 EXCHANGE_FILTER(ch));
			free(EXCHANGE_FILTER(ch));
			EXCHANGE_FILTER(ch) = nullptr;
		} else {
			sprintf(tmpbuf, "  .\r\n");
		}
		SendMsgToChar(tmpbuf, ch);
		return true;
	}

	ParseFilter params(ParseFilter::EXCHANGE);
	if (!parse_exch_filter(params, filter, false)) {
		SendMsgToChar("  .  .\r\n", ch);
		free(EXCHANGE_FILTER(ch));
		EXCHANGE_FILTER(ch) = nullptr;
		return false;
	}

	if (EXCHANGE_FILTER(ch)) {
		snprintf(tmpbuf, kMaxInputLength, "  : %s.  : %s.\r\n",
				 EXCHANGE_FILTER(ch), filter);
	} else {
		snprintf(tmpbuf, kMaxInputLength, "  : %s.\r\n", filter);
	}
	SendMsgToChar(tmpbuf, ch);

	if (EXCHANGE_FILTER(ch))
		free(EXCHANGE_FILTER(ch));
	EXCHANGE_FILTER(ch) = str_dup(filter);

	return true;
}

ExchangeItem *create_exchange_item(void) {
	ExchangeItem *item;
	CREATE(item, 1);
	GET_EXCHANGE_ITEM_LOT(item) = -1;
	GET_EXCHANGE_ITEM_SELLERID(item) = -1;
	GET_EXCHANGE_ITEM_COST(item) = 0;
	item->time = time(nullptr);
	GET_EXCHANGE_ITEM_COMMENT(item) = nullptr;
	GET_EXCHANGE_ITEM(item) = nullptr;
	item->next = exchange_item_list;
	exchange_item_list = item;
	return (item);
}

void extract_exchange_item(ExchangeItem *item) {
	REMOVE_FROM_LIST(item, exchange_item_list);
	if (GET_EXCHANGE_ITEM_LOT(item) > 0 && GET_EXCHANGE_ITEM_LOT(item) <= (int) lot_usage.size())
		lot_usage[GET_EXCHANGE_ITEM_LOT(item) - 1] = false;
	if (item->comment)
		free(item->comment);
	if (item->obj)
		ExtractObjFromWorld(item->obj);
	free(item);

}

void check_exchange(ObjData *obj) {
	if (!obj) {
		return;
	}
	ExchangeItem *j, *next_thing;

	for (j = exchange_item_list; j; j = next_thing) {
		next_thing = j->next;
		if (GET_EXCHANGE_ITEM(j) == obj) {
			REMOVE_FROM_LIST(j, exchange_item_list);
			lot_usage[GET_EXCHANGE_ITEM_LOT(j) - 1] = false;
			if (j->comment) {
				free(j->comment);
			}
			free(j);
			break;
		}
	}
}

ExchangeItem *exchange_read_one_object_new(char **data, int *error) {
	char buffer[kMaxStringLength];
	ExchangeItem *item = nullptr;
	//char *d;

	*error = 1;
	//    
	for (; **data != EX_NEW_ITEM_CHAR; (*data)++)
		if (!**data || **data == EX_END_CHAR)
			return (item);

	*error = 2;

	//  #
	(*data)++;

	//  lot 
	if (!get_buf_line(data, buffer))
		return (item);

	*error = 3;
	item = create_exchange_item();
	if ((GET_EXCHANGE_ITEM_LOT(item) = atoi(buffer)) <= 0)
		return (item);

	*error = 4;
	//  seler_id 
	if (!get_buf_line(data, buffer))
		return (item);
	*error = 5;
	if (!(GET_EXCHANGE_ITEM_SELLERID(item) = atoi(buffer)))
		return (item);
	*error = 6;
	//   
	if (!get_buf_line(data, buffer))
		return (item);

	*error = 7;
	if (!(GET_EXCHANGE_ITEM_COST(item) = atoi(buffer)))
		return (item);

	if (!get_buf_line(data, buffer))
		return (item);
	*error = 8;
	if (!(item->time = atoi(buffer)))
		return (item);
	*error = 9;
	//  comment 
	char *str_last_symb = strchr(*data, '\n');
	strncpy(buffer, *data, str_last_symb - *data);
	buffer[str_last_symb - *data] = '\0';
	*data = str_last_symb;

	if (strcmp(buffer, "EMPTY"))
		if (!(GET_EXCHANGE_ITEM_COMMENT(item) = str_dup(buffer)))
			return (item);

	*error = 0;

	GET_EXCHANGE_ITEM(item) = read_one_object_new(data, error).get();
	if (*error) {
		*error = 9;
	}

	return (item);
}

void exchange_write_one_object_new(std::stringstream &out, ExchangeItem *item) {
	out << "#" << GET_EXCHANGE_ITEM_LOT(item) << "\n"
		<< GET_EXCHANGE_ITEM_SELLERID(item) << "\n"
		<< GET_EXCHANGE_ITEM_COST(item) << "\n"
		<< item->time << "\n"
		<< (GET_EXCHANGE_ITEM_COMMENT(item) ? GET_EXCHANGE_ITEM_COMMENT(item) : "EMPTY\n")
		<< "\n";
	write_one_object(out, GET_EXCHANGE_ITEM(item), 0);
	out << "\n";
}

int exchange_database_load() {
	FILE *fl;
	char *data, *readdata;
	ExchangeItem *item;
	int fsize, error, max_lot = 0;
	char buffer[kMaxStringLength];

	log("Exchange: loading database... (exchange.cpp)");

	if (!(fl = fopen(EXCHANGE_DATABASE_FILE, "r"))) {
		log("SYSERR: Error opening exchange database. (exchange.cpp)");
		return (0);
	}

	fseek(fl, 0L, SEEK_END);
	fsize = ftell(fl);

	CREATE(readdata, fsize + 1);
	fseek(fl, 0L, SEEK_SET);
	auto actual_size = fread(readdata, 1, fsize, fl);
	if (!actual_size || ferror(fl)) {
		fclose(fl);
		log("SYSERR: Memory error or cann't read exchange database file. (exchange.cpp)");
		free(readdata);
		return (0);
	};
	fclose(fl);

	data = readdata;
	*(data + actual_size) = '\0';

	//    ?
	get_buf_line(&data, buffer);
	if (strstr(buffer, "!NEW!") == nullptr) {
		log("SYSERR: Old format exchange database file.");
		return 0;
	}

	for (fsize = 0; *data && *data != EX_END_CHAR; fsize++) {
		item = exchange_read_one_object_new(&data, &error);

		if (item == nullptr) {
			log("SYSERR: Error #%d reading exchange database file. (exchange.cpp)", error);
			return (0);
		}

		if (error) {
			log("SYSERR: Error #%d reading item from exchange database.", error);
			extract_exchange_item(item);
			continue;
		}

		//    
		if (GET_EXCHANGE_ITEM(item)->get_timer() <= 0) {
			std::string cap = GET_EXCHANGE_ITEM(item)->get_PName(0);
			cap[0] = UPPER(cap[0]);
			log("Exchange: - %s %s   .\r\n",
				cap.c_str(), GET_OBJ_SUF_2(GET_EXCHANGE_ITEM(item)));
			extract_exchange_item(item);
			continue;
		}
		max_lot = std::max(max_lot, GET_EXCHANGE_ITEM_LOT(item));
	}

	free(readdata);

	lot_usage.resize(max_lot, false);
	for (item = exchange_item_list; item; item = item->next)
		lot_usage[GET_EXCHANGE_ITEM_LOT(item) - 1] = true;

	log("Exchange: done loading database.");

	return (1);
}

int exchange_database_reload(bool loadbackup) {
	FILE *fl;
	char *data, *readdata;
	ExchangeItem *item, *j, *next_thing;
	int fsize, error, max_lot = 0;
	char buffer[kMaxStringLength];

	for (j = exchange_item_list; j; j = next_thing) {
		next_thing = j->next;
		extract_exchange_item(j);
	}
	if (loadbackup) {
		log("Exchange: reloading backup of database... (exchange.cpp)");
		if (!(fl = fopen(EXCHANGE_DATABASE_BACKUPFILE, "r"))) {
			log("SYSERR: Error opening exchange database backup. (exchange.cpp)");
			return (0);
		}
	} else {
		log("Exchange: reloading database... (exchange.cpp)");
		if (!(fl = fopen(EXCHANGE_DATABASE_FILE, "r"))) {
			log("SYSERR: Error opening exchange database. (exchange.cpp)");
			return (0);
		}
	}

	fseek(fl, 0L, SEEK_END);
	fsize = ftell(fl);

	CREATE(readdata, fsize + 1);
	fseek(fl, 0L, SEEK_SET);
	auto actual_size = fread(readdata, 1, fsize, fl);
	if (!actual_size || ferror(fl)) {
		fclose(fl);
		if (loadbackup)
			log("SYSERR: Memory error or cann't read exchange database backup file. (exchange.cpp)");
		else
			log("SYSERR: Memory error or cann't read exchange database file. (exchange.cpp)");
		free(readdata);
		return (0);
	};
	fclose(fl);

	data = readdata;
	*(data + actual_size) = '\0';

	//    ?
	get_buf_line(&data, buffer);
	if (strstr(buffer, "!NEW!") == nullptr) {
		log("SYSERR: Old format exchange database file.");
		return 0;
	}

	for (fsize = 0; *data && *data != EX_END_CHAR; fsize++) {
		item = exchange_read_one_object_new(&data, &error);

		if (item == nullptr) {
			if (loadbackup)
				log("SYSERR: Error #%d reading exchange database backup file. (exchange.cpp)", error);
			else
				log("SYSERR: Error #%d reading exchange database file. (exchange.cpp)", error);
			return (0);
		}

		if (error) {
			if (loadbackup)
				log("SYSERR: Error #%d reading item from exchange database backup.", error);
			else
				log("SYSERR: Error #%d reading item from exchange database.", error);
			extract_exchange_item(item);
			continue;
		}

		//    
		if (GET_EXCHANGE_ITEM(item)->get_timer() <= 0) {
			std::string cap = GET_EXCHANGE_ITEM(item)->get_PName(0);
			cap[0] = UPPER(cap[0]);
			log("Exchange: - %s %s   .\r\n",
				cap.c_str(), GET_OBJ_SUF_2(GET_EXCHANGE_ITEM(item)));
			extract_exchange_item(item);
			continue;
		}
		max_lot = std::max(max_lot, GET_EXCHANGE_ITEM_LOT(item));
	}

	free(readdata);

	lot_usage.resize(max_lot, false);
	for (item = exchange_item_list; item; item = item->next)
		lot_usage[GET_EXCHANGE_ITEM_LOT(item) - 1] = true;

	if (loadbackup)
		log("Exchange: done reloading database backup.");
	else
		log("Exchange: done reloading database.");

	return (1);
}

/**
* \param backup - false ()  
*               - true   ,   
*/
void exchange_database_save(bool backup) {
	log("Exchange: Saving exchange database...");

	std::stringstream out;
	out << "!NEW!\n";
	ExchangeItem *j, *next_thing;
	for (j = exchange_item_list; j; j = next_thing) {
		next_thing = j->next;
		exchange_write_one_object_new(out, j);
	}
	out << "$\n$\n";

	const char *filename = backup ? EXCHANGE_DATABASE_BACKUPFILE : EXCHANGE_DATABASE_FILE;
	std::ofstream file(filename);
	if (!file.is_open()) {
		sprintf(buf, "[SYSERROR] open exchange db ('%s')", filename);
		mudlog(buf, BRF, kLvlImmortal, SYSLOG, true);
		return;
	}
	file << out.rdbuf();
	file.close();

	log("Exchange: done saving database.");
	return;
}

int get_unique_lot(void) {
	int i;
	for (i = 0; i < (int) lot_usage.size(); i++)
		if (!lot_usage[i]) {
			lot_usage[i] = true;
			return i + 1;
		}

	if (lot_usage.size() < INT_MAX) {
		lot_usage.push_back(true);
		return i + 1;
	} else
		return -1;
}

void message_exchange(char *message, CharData *ch, ExchangeItem *j) {
	for (DescriptorData *i = descriptor_list; i; i = i->next) {
		if (STATE(i) == CON_PLAYING
			&& (!ch || i != ch->desc)
			&& i->character
			&& !PRF_FLAGGED(i->character, EPrf::kNoExchange)
			&& !PLR_FLAGGED(i->character, EPlrFlag::kWriting)
			&& !ROOM_FLAGGED(IN_ROOM(i->character), ERoomFlag::kSoundproof)
			&& GET_POS(i->character) > EPosition::kSleep) {
			if (!PRF_FLAGGED(i->character, EPrf::kNoIngrMode)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kIngredient
					|| GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kMagicIngredient)) {
				continue;
			}
			ParseFilter params(ParseFilter::EXCHANGE);
			if (!EXCHANGE_FILTER(i->character)
				|| (parse_exch_filter(params, EXCHANGE_FILTER(i->character), false)
					&& params.check(j))) {
				if (COLOR_LEV(i->character) >= C_NRM) {
					SendMsgToChar("&Y&q", i->character.get());
				}

				act(message, false, i->character.get(), 0, 0, kToChar | kToSleep);

				if (COLOR_LEV(i->character) >= C_NRM) {
					SendMsgToChar("&Q&n", i->character.get());
				}
			}
		}
	}
}

void show_lots(char *filter, short int show_type, CharData *ch) {
	/*
	show_type
	0 - 
	1 - 
	2 - 
	3 - 
	4 - 
	5 - 
	6 - 
	7 - 
	8 -  
	9 -  
	10 -  
	11 - 
	*/
	char tmpbuf[kMaxInputLength];
	bool any_item = 0;

	ParseFilter params(ParseFilter::EXCHANGE);
	if (!parse_exch_filter(params, filter, true)) {
		SendMsgToChar("  !\r\n", ch);
		log("Exchange: Player uses wrong filter '%s'", filter);
		return;
	}

	std::string buffer;
	if (show_type == 11) {
		buffer += params.print();
	}
	buffer +=
		"                                                             \r\n"
		"--------------------------------------------------------------------------------------------\r\n";

	for (ExchangeItem *j = exchange_item_list; j; j = j->next) {
		if (!params.check(j)
			|| ((show_type == 1)
				&& (!isname(GET_NAME(ch), get_name_by_id(GET_EXCHANGE_ITEM_SELLERID(j)))))
			|| ((show_type == 2)
				&& ((GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kIngredient)
					|| (GET_OBJ_VNUM(GET_EXCHANGE_ITEM(j)) < 200)
					|| (GET_OBJ_VNUM(GET_EXCHANGE_ITEM(j)) > 299)))
			|| ((show_type == 3)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kArmor)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kLightArmor)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kMediumArmor)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kHeavyArmor))
			|| ((show_type == 4)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kWeapon))
			|| ((show_type == 5)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kBook))
			|| (show_type == 6
				&& GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kMagicIngredient
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kIngredient
					|| (GET_OBJ_VNUM(GET_EXCHANGE_ITEM(j)) >= 200
						&& GET_OBJ_VNUM(GET_EXCHANGE_ITEM(j)) <= 299)))
			|| ((show_type == 7)
				&& ((GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kIngredient)
					|| ObjSystem::is_armor_type(GET_EXCHANGE_ITEM(j))
					|| (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kWeapon)
					|| (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kBook)
					|| (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) == EObjType::kMagicIngredient)))
			|| ((show_type == 8)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kLightArmor))
			|| ((show_type == 9)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kMediumArmor))
			|| ((show_type == 10)
				&& (GET_OBJ_TYPE(GET_EXCHANGE_ITEM(j)) != EObjType::kHeavyArmor))) {
			continue;
		}
		//     5-10  
		if (utils::IsAbbr(" ", GET_EXCHANGE_ITEM(j)->get_PName(0).c_str())
			|| utils::IsAbbr("  ", GET_EXCHANGE_ITEM(j)->get_PName(0).c_str())
			|| utils::IsAbbr(" ", GET_EXCHANGE_ITEM(j)->get_PName(0).c_str())) {
			GET_EXCHANGE_ITEM(j)->get_affect_flags().sprintbits(weapon_affects, buf, ",");
			//   ,        
			if (!strcmp(buf,
						""))  // added by WorM ()    ,   . 
			{
				bool found = false;
				for (int n = 0; n < kMaxObjAffect; n++) {
					auto drndice = GET_EXCHANGE_ITEM(j)->get_affected(n).location;
					int drsdice = GET_EXCHANGE_ITEM(j)->get_affected(n).modifier;
					if ((drndice != EApply::kNone) && (drsdice != 0)) {
						bool negative = IsNegativeApply(drndice);
						if (drsdice < 0)
							negative = !negative;
						sprinttype(drndice, apply_types, buf2);
						snprintf(buf, kMaxStringLength, "%s %s%d", buf2, negative ? "-" : "+", abs(drsdice));
						found = true;
						break;
					}
				}

				if (!found) {
					sprintf(tmpbuf,
							"[%4d]   %s",
							GET_EXCHANGE_ITEM_LOT(j),
							GET_OBJ_PNAME(GET_EXCHANGE_ITEM(j), 0).c_str());
				} else {
					snprintf(tmpbuf,
							 kMaxInputLength,
							 "[%4d]   %s (%s)",
							 GET_EXCHANGE_ITEM_LOT(j),
							 GET_OBJ_PNAME(GET_EXCHANGE_ITEM(j), 0).c_str(),
							 buf);
				}
			} else  // end by WorM
			{
				snprintf(tmpbuf, kMaxInputLength, "[%4d]   %s (%s)", GET_EXCHANGE_ITEM_LOT(j),
						 GET_OBJ_PNAME(GET_EXCHANGE_ITEM(j), 0).c_str(), buf);
			}
		} else if (is_dig_stone(GET_EXCHANGE_ITEM(j))
			&& GET_OBJ_MATER(GET_EXCHANGE_ITEM(j)) == EObjMaterial::kGlass) {
			sprintf(tmpbuf,
					"[%4d]   %s ()",
					GET_EXCHANGE_ITEM_LOT(j),
					GET_OBJ_PNAME(GET_EXCHANGE_ITEM(j), 0).c_str());
		} else {
			sprintf(tmpbuf,
					"[%4d]   %s%s",
					GET_EXCHANGE_ITEM_LOT(j),
					GET_OBJ_PNAME(GET_EXCHANGE_ITEM(j), 0).c_str(),
					params.show_obj_aff(GET_EXCHANGE_ITEM(j)).c_str());
		}
		char *tmstr;
		tmstr = (char *) asctime(localtime(&(j->time)));
		if (IS_GOD(ch)) //asctime    
			sprintf(tmpbuf,
					"%s %9d  %-s %s",
					colored_name(tmpbuf, 63, true),
					GET_EXCHANGE_ITEM_COST(j),
					diag_obj_timer(GET_EXCHANGE_ITEM(j)),
					tmstr);
		else
			sprintf(tmpbuf,
					"%s %9d  %-s\r\n",
					colored_name(tmpbuf, 63, true),
					GET_EXCHANGE_ITEM_COST(j),
					diag_obj_timer(GET_EXCHANGE_ITEM(j)));
		//   ,         0.6     .   -- Krodo
		buffer += tmpbuf;
		any_item = 1;
	}

	if (!any_item) {
		buffer = " !\r\n";
	} else if (show_type == 11) {
		const int price = EXCHANGE_IDENT_PAY / 2;
		ch->remove_both_gold(price);
		SendMsgToChar(ch, "\r\n%s         %d %s%s\r\n",
					  CCIGRN(ch, C_NRM), price, GetDeclensionInNumber(price, EWhat::kMoneyU), CCNRM(ch, C_NRM));
	}
	page_string(ch->desc, buffer);
}

int parse_exch_filter(ParseFilter &filter, char *buf, bool parse_affects) {
	char tmpbuf[FILTER_LENGTH];

	while (*buf && (*buf != '\r') && (*buf != '\n')) {
		switch (*buf) {
			case ' ': buf++;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				filter.name = tmpbuf;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				filter.owner = tmpbuf;
				//filter.owner_id = get_id_by_name(tmpbuf);
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_type(tmpbuf))
					return 0;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_cost(tmpbuf))
					return 0;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_state(tmpbuf))
					return 0;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_wear(tmpbuf))
					return 0;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_weap_class(tmpbuf))
					return 0;
				break;
			case '':
				if (!parse_affects)
					return 0;
				buf = one_argument(++buf, tmpbuf);
				if (filter.affects_cnt() >= 1)
					break;
				else if (!filter.init_affect(tmpbuf, strlen(tmpbuf)))
					return 0;
				break;
			case '': buf = one_argument(++buf, tmpbuf);
				if (!filter.init_realtime(tmpbuf))
					return 0;
				break;
            case '': buf = one_argument(++buf, tmpbuf);
                if (!filter.init_remorts(tmpbuf))
                    return 0;
                break;
			default: return 0;
		}
	}

	return 1;

}

void clear_exchange_lot(ExchangeItem *lot) {
	if (lot == nullptr) {
		return;
	}

	REMOVE_FROM_LIST(lot, exchange_item_list);
	lot_usage[GET_EXCHANGE_ITEM_LOT(lot) - 1] = false;
	if (lot->comment) {
		free(lot->comment);
	}
	free(lot);
}

//Polud   ,          
void do_exchange(CharData *ch, char *argument, int cmd, int/* subcmd*/) {
	char *arg = str_dup(argument);
	argument = one_argument(argument, arg1);

	if (ch->IsNpc()) {
		SendMsgToChar("?!     !\r\n", ch);
	} else if ((utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "exhibit")
		|| utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "cost")
		|| utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "withdraw")
		|| utils::IsAbbr(arg1, "") || utils::IsAbbr(arg1, "purchase")) && !IS_IMPL(ch)) {
		SendMsgToChar("     ,    .\r\n",
					 ch);
	} else
		exchange(ch, nullptr, cmd, arg);
	free(arg);
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
