// title.cpp
// Copyright (c) 2006 Krodo
// Part of Bylins http://www.bylins.su

#include "title.h"

#include <boost/algorithm/string.hpp>

#include "entities/char_player.h"
#include "game_fight/pk.h"
#include "handler.h"
#include "administration/privilege.h"
#include "color.h"

extern void send_to_gods(char *text, bool demigod);

namespace TitleSystem {

//   ,      
struct waiting_title {
	std::string title;     // 
	std::string pre_title; // 
	long unique;           //   (  )
};

const unsigned int MAX_TITLE_LENGTH = 80; // .   (+)
const int SET_TITLE_COST = 1000;          //     
const char *TITLE_FILE = LIB_PLRSTUFF"titles.lst"; //  /   
const char *MORTAL_DO_TITLE_FORMAT =
	" -       ,       \r\n"
	"  <> -    ,  \r\n"
	"  < ! > - ,    \r\n"
	"  -     \r\n"
	"  -        \r\n"
	"  -       (  )\r\n";
const char *GOD_DO_TITLE_FORMAT =
	" -          \r\n"
	" <> || -        ,     \r\n"
	"  < > -   \r\n"
	"  < ! > -     \r\n"
	"  -       (  )\r\n";
const char
	*TITLE_SEND_FORMAT = "    ,  ,    ' '\r\n";

enum { TITLE_FIND_CHAR = 0, TITLE_CANT_FIND_CHAR, TITLE_NEED_HELP };
typedef std::shared_ptr<waiting_title> WaitingTitlePtr;
typedef std::map<std::string, WaitingTitlePtr> TitleListType;
TitleListType title_list;
TitleListType temp_title_list;

bool check_title(const std::string &text, CharData *ch);
bool check_pre_title(std::string text, CharData *ch);
bool check_alphabet(const std::string &text, CharData *ch, const std::string &allowed);
bool is_new_petition(CharData *ch);
bool manage_title_list(std::string &name, bool action, CharData *ch);
void set_player_title(CharData *ch, const std::string &pre_title, const std::string &title, const char *god);
const char *print_help_string(CharData *ch);
std::string print_agree_string(bool new_petittion);
std::string print_title_string(CharData *ch, const std::string &pre_title, const std::string &title);
std::string print_title_string(const std::string &name, const std::string &pre_title, const std::string &title);
void do_title_empty(CharData *ch);
DescriptorData *send_result_message(long unique, bool action);

} // namespace TitleSystem

// *  , title. ACMD(do_title),        .
void TitleSystem::do_title(CharData *ch, char *argument, int/* cmd*/, int/* subcmd*/) {
	if (ch->IsNpc()) return;

	if (!privilege::CheckFlag(ch, privilege::kTitle) && PLR_FLAGGED(ch, EPlrFlag::kNoTitle)) {
		SendMsgToChar("    .\r\n", ch);
		return;
	}

	std::string buffer(argument), buffer2;
	GetOneParam(buffer, buffer2);

	if (buffer2.empty()) {
		do_title_empty(ch);
		return;
	}

	//      ,     
	int result = TITLE_NEED_HELP;
	if (IS_GOD(ch) || privilege::CheckFlag(ch, privilege::kTitle)) {
		utils::Trim(buffer);
		if (CompareParam(buffer, "")) {
			utils::Trim(buffer2);
			CharData *vict = get_player_pun(ch, buffer2, EFind::kCharInWorld);
			if (!vict) {
				SendMsgToChar("  .\r\n", ch);
				return;
			}
			if (GetRealLevel(vict) >= kLvlImmortal || PRF_FLAGGED(vict, EPrf::kCoderinfo)) {
				SendMsgToChar("    .\r\n", ch);
				return;
			}
			if (GET_TITLE(vict) != "") {
				sprintf(buf, "&c%s    %s.&n\r\n", GET_NAME(ch), GET_NAME(vict));
				send_to_gods(buf, true);
				GET_TITLE(vict) = "";
				//SendMsgToChar(" .\r\n", ch);
			} else
				SendMsgToChar("   .\r\n", ch);
			return;
		} else if (CompareParam(buffer, ""))
			result = manage_title_list(buffer2, 1, ch);
		else if (CompareParam(buffer, ""))
			result = manage_title_list(buffer2, 0, ch);
		if (result == TITLE_FIND_CHAR)
			return;
	}

	if (CompareParam(buffer2, "")) {
		utils::Trim(buffer);
		if (buffer.size() > MAX_TITLE_LENGTH) {
			if (PLR_FLAGGED(ch, EPlrFlag::kNoTitle)) {
				SendMsgToChar(ch, "     .\r\n");
				return;
			}
			SendMsgToChar(ch, "     %d .\r\n", MAX_TITLE_LENGTH);
			return;
		}

		std::string pre_title, title;
		std::string::size_type beg_idx = buffer.find('!');
		if (beg_idx != std::string::npos) {
			pre_title = buffer.substr(0, beg_idx);
			beg_idx = buffer.find_first_not_of('!', beg_idx);
			if (beg_idx != std::string::npos)
				title = buffer.substr(beg_idx);
			utils::Trim(title);
			utils::Trim(pre_title);
			if (!pre_title.empty()) {
				sprintf(buf2, "%c%s", UPPER(pre_title.substr(0, 1).c_str()[0]), pre_title.substr(1).c_str());
				pre_title = buf2;
			}
			if (!pre_title.empty() && !check_pre_title(pre_title, ch)) return;
			if (!title.empty() && !check_title(title, ch)) return;
		} else {
			if (!check_title(buffer, ch)) return;
			title = buffer;
		}
		if (IS_GOD(ch) || privilege::CheckFlag(ch, privilege::kTitle)) {
			set_player_title(ch, pre_title, title, GET_NAME(ch));
			SendMsgToChar(" \r\n", ch);
			return;
		}
		if (title.empty() && pre_title.empty()) {
			SendMsgToChar("      .\r\n", ch);
			return;
		}

		bool new_petition = is_new_petition(ch);
		WaitingTitlePtr temp(new waiting_title);
		temp->title = title;
		temp->pre_title = pre_title;
		temp->unique = GET_UNIQUE(ch);
		temp_title_list[GET_NAME(ch)] = temp;

		std::stringstream out;
		out << "     :\r\n" << CCPK(ch, C_NRM, ch);
		out << print_title_string(ch, pre_title, title) << print_agree_string(new_petition);
		SendMsgToChar(out.str(), ch);
	} else if (CompareParam(buffer2, "")) {
		TitleListType::iterator it = temp_title_list.find(GET_NAME(ch));
		if (it != temp_title_list.end()) {
			if (ch->get_bank() < SET_TITLE_COST) {
				SendMsgToChar("         .\r\n", ch);
				return;
			}
			ch->remove_bank(SET_TITLE_COST);
			title_list[it->first] = it->second;
			temp_title_list.erase(it);
			SendMsgToChar("         .\r\n", ch);
			SendMsgToChar(TITLE_SEND_FORMAT, ch);
		} else
			SendMsgToChar("    ,     .\r\n", ch);
	} else if (CompareParam(buffer2, "")) {
		TitleListType::iterator it = title_list.find(GET_NAME(ch));
		if (it != title_list.end()) {
			title_list.erase(it);
			ch->add_bank(SET_TITLE_COST);
			SendMsgToChar("    .\r\n", ch);
		} else
			SendMsgToChar("     .\r\n", ch);
	} else if (CompareParam(buffer2, "", 1)) {
		if (GET_TITLE(ch) != "") {
			GET_TITLE(ch) = "";
		}
		SendMsgToChar("    .\r\n", ch);
	} else {
		//  ,       ,     ,   
		if (result == TITLE_CANT_FIND_CHAR)
			SendMsgToChar("      .\r\n", ch);
		else if (result == TITLE_NEED_HELP)
			SendMsgToChar(print_help_string(ch), ch);
	}
}

/**
*      
*   25+    ,   
* \return 0  , 1 
*/
bool TitleSystem::check_title(const std::string &text, CharData *ch) {
	if (!check_alphabet(text, ch, " ,.-?")) return 0;

	if (GetRealLevel(ch) < 25 && !GetRealRemort(ch) && !IS_GOD(ch) && !privilege::CheckFlag(ch, privilege::kTitle)) {
		SendMsgToChar(ch, "       25    .\r\n");
		return 0;
	}

	return 1;
}

/**
*      .
*         ( -  4+ )  ,
*  3  -   .   3   3 ,   .
* \return 0  , 1 
*/
bool TitleSystem::check_pre_title(std::string text, CharData *ch) {
	if (!check_alphabet(text, ch, " .-?")) return 0;

	if (IS_GOD(ch) || privilege::CheckFlag(ch, privilege::kTitle)) return 1;

	if (!GetRealRemort(ch)) {
		SendMsgToChar(ch, "         .\r\n");
		return 0;
	}

	int word = 0, prep = 0;
	typedef boost::split_iterator<std::string::iterator> split_it;
	for (split_it it = boost::make_split_iterator(text, boost::first_finder(" ", boost::is_iequal())); it != split_it();
		 ++it) {
		if (boost::copy_range<std::string>(*it).size() > 3)
			++word;
		else
			++prep;
	}
	if (word > 3 || prep > 3) {
		SendMsgToChar(ch, "    .\r\n");
		return 0;
	}
	if (word > GetRealRemort(ch)) {
		SendMsgToChar(ch, "        .\r\n");
		return 0;
	}

	return 1;
}

/**
*                
* \param allowed -        (    )
* \return 0  , 1 
*/
bool TitleSystem::check_alphabet(const std::string &text, CharData *ch, const std::string &allowed) {
	int i = 0;
	std::string::size_type idx;
	for (std::string::const_iterator it = text.begin(); it != text.end(); ++it, ++i) {
		unsigned char c = static_cast<char>(*it);
		idx = allowed.find(*it);
		if (c < 192 && idx == std::string::npos) {
			SendMsgToChar(ch, "  '%c'   %d.\r\n", *it, ++i);
			return 0;
		}
	}
	return 1;
}

/**
*      ,   
* \param unique -  ,   
* \param action - 0  , 1  
*/
DescriptorData *TitleSystem::send_result_message(long unique, bool action) {
	DescriptorData *d = DescByUID(unique);
	if (d) {
		SendMsgToChar(d->character.get(), "   %s .\r\n", action ? "" : "");
	}
	return d;
}

/**
*       /.
*    ,   .   ,  .
* \param action - 0  , 1 
*/
bool TitleSystem::manage_title_list(std::string &name, bool action, CharData *ch) {
	name_convert(name);
	TitleListType::iterator it = title_list.find(name);
	if (it != title_list.end()) {
		if (action) {
			DescriptorData *d = send_result_message(it->second->unique, action);
			//     ?
			if (d) {
				set_player_title(d->character.get(), it->second->pre_title, it->second->title, GET_NAME(ch));
				sprintf(buf, "&c%s    %s!&n\r\n", GET_NAME(ch), GET_NAME(d->character));
				send_to_gods(buf, true);
			} else {
				Player victim;
				if (load_char(it->first.c_str(), &victim) < 0) {
					SendMsgToChar("     - .\r\n", ch);
					title_list.erase(it);
					return TITLE_FIND_CHAR;
				}
				set_player_title(&victim, it->second->pre_title, it->second->title, GET_NAME(ch));
				sprintf(buf, "&c%s    %s[].&n\r\n", GET_NAME(ch), GET_NAME(&victim));
				send_to_gods(buf, true);
				victim.save_char();
			}
		} else {
			send_result_message(it->second->unique, action);

			DescriptorData *d = send_result_message(it->second->unique, action);
			if (d) {
				sprintf(buf, "&c%s    %s.&n\r\n", GET_NAME(ch), GET_NAME(d->character));
				send_to_gods(buf, true);
			} else {
				Player victim;
				if (load_char(it->first.c_str(), &victim) < 0) {
					SendMsgToChar("     - .\r\n", ch);
					title_list.erase(it);
					return TITLE_FIND_CHAR;
				}
				sprintf(buf, "&c%s    %s[].&n\r\n", GET_NAME(ch), GET_NAME(&victim));
				send_to_gods(buf, true);
				victim.save_char();
			}
		}
		title_list.erase(it);
		return TITLE_FIND_CHAR;
	}
	return TITLE_CANT_FIND_CHAR;
}

// *   ,  
bool TitleSystem::show_title_list(CharData *ch) {
	if (title_list.empty())
		return false;

	std::stringstream out;
	out << "\r\n     ( <> /):\r\n" << CCWHT(ch, C_NRM);
	for (TitleListType::const_iterator it = title_list.begin(); it != title_list.end(); ++it)
		out << print_title_string(it->first, it->second->pre_title, it->second->title);
	out << CCNRM(ch, C_NRM);
	SendMsgToChar(out.str(), ch);
	return true;
}

// *       (  )
std::string TitleSystem::print_title_string(const std::string &name,
											const std::string &pre_title,
											const std::string &title) {
	std::stringstream out;
	if (!pre_title.empty())
		out << pre_title << " ";
	out << name;
	if (!title.empty())
		out << ", " << title;
	out << "\r\n";
	return out.str();
}

// *   ,          
std::string TitleSystem::print_title_string(CharData *ch, const std::string &pre_title, const std::string &title) {
	std::stringstream out;
	out << CCPK(ch, C_NRM, ch);
	if (!pre_title.empty())
		out << pre_title << " ";
	out << GET_NAME(ch);
	if (!title.empty())
		out << ", " << title;
	out << CCNRM(ch, C_NRM) << "\r\n";
	return out.str();
}

/**
*    (GET_TITLE(ch)) 
* \param ch -  
* \param god -    
*/
void TitleSystem::set_player_title(CharData *ch,
								   const std::string &pre_title,
								   const std::string &title,
								   const char *god) {
	std::stringstream out;
	out << title;
	if (!pre_title.empty())
		out << ";" << pre_title;
	out << "/ : " << god << "/";
	GET_TITLE(ch) = out.str();
}

// *       
const char *TitleSystem::print_help_string(CharData *ch) {
	if (IS_GOD(ch) || privilege::CheckFlag(ch, privilege::kTitle))
		return GOD_DO_TITLE_FORMAT;

	return MORTAL_DO_TITLE_FORMAT;
}

// *     
void TitleSystem::save_title_list() {
	std::ofstream file(TITLE_FILE);
	if (!file.is_open()) {
		log("Error open file: %s! (%s %s %d)", TITLE_FILE, __FILE__, __func__, __LINE__);
		return;
	}
	for (TitleListType::const_iterator it = title_list.begin(); it != title_list.end(); ++it)
		file << it->first << " " << it->second->unique << "\n" << it->second->pre_title << "\n" << it->second->title
			 << "\n";
	file.close();
}

// *     ,     reload title
void TitleSystem::load_title_list() {
	title_list.clear();

	std::ifstream file(TITLE_FILE);
	if (!file.is_open()) {
		log("Error open file: %s! (%s %s %d)", TITLE_FILE, __FILE__, __func__, __LINE__);
		return;
	}
	std::string name, pre_title, title;
	long unique;
	while (file >> name >> unique) {
		ReadEndString(file);
		std::getline(file, pre_title);
		std::getline(file, title);
		WaitingTitlePtr temp(new waiting_title);
		temp->title = title;
		temp->pre_title = pre_title;
		temp->unique = unique;
		title_list[name] = temp;
	}
	file.close();
}

/**
*          ( ,     )
* \return 0        , 1  
*/
bool TitleSystem::is_new_petition(CharData *ch) {
	TitleListType::iterator it = temp_title_list.find(GET_NAME(ch));
	if (it != temp_title_list.end())
		return 0;
	return 1;
}

/**
*          
* \param new_petition - 0   , 1  
*/
std::string TitleSystem::print_agree_string(bool new_petition) {
	std::stringstream out;
	out << "    ' '";
	if (new_petition)
		out << ",   " << SET_TITLE_COST << " .";
	out << "\r\n";
	return out.str();
}

// *     
void TitleSystem::do_title_empty(CharData *ch) {
	if ((IS_GOD(ch) || privilege::CheckFlag(ch, privilege::kTitle)) && !title_list.empty())
		show_title_list(ch);
	else {
		std::stringstream out;
		TitleListType::iterator it = title_list.find(GET_NAME(ch));
		if (it != title_list.end()) {
			out << "      :\r\n"
				<< print_title_string(ch, it->second->pre_title, it->second->title)
				<< TITLE_SEND_FORMAT << "\r\n";
		}
		it = temp_title_list.find(GET_NAME(ch));
		if (it != temp_title_list.end()) {
			out << "        :\r\n"
				<< print_title_string(ch, it->second->pre_title, it->second->title)
				<< print_agree_string(is_new_petition(ch)) << "\r\n";
		}
		out << print_help_string(ch);
		SendMsgToChar(out.str(), ch);
	}
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
