/*
 \authors Created by Sventovit
 \date 04.06.2021.
 \brief    .
 \details -    .       ,
   ,       .      
     ,   .      
  .

  .      DataNode,      
    key-value.  - ,     -  DataNode, ,
   ,  ,  , .     ,   
        ,     .   xml (
  -   xml),    .   ,      
 ,    ,
 */

#ifndef PARSER_WRAPPER
#define PARSER_WRAPPER

#include <filesystem>
#include <memory>

#include "third_party_libs/pugixml/pugixml.h"
#include "structs/iterators.h"

namespace parser_wrapper {

class DataNode : public std::iterator<
	std::forward_iterator_tag,  // iterator_category
	std::ptrdiff_t,             // difference
	DataNode,                   // value_type
	DataNode *,                 // pointer
	DataNode &                  // reference
> {
 protected:
	using DocPtr = std::shared_ptr<pugi::xml_document>;

	DocPtr xml_doc_;
	//        .
	pugi::xml_node curren_xml_node_;

 public:
	DataNode() :
		xml_doc_{std::make_shared<pugi::xml_document>()},
		curren_xml_node_{pugi::xml_node()} {};

	explicit DataNode(const std::filesystem::path &file_name);

	DataNode(const DataNode &d);

	~DataNode() = default;

	class NameIterator {
		using iterator_category = std::forward_iterator_tag;
		using difference_type   = std::ptrdiff_t;
		using value_type        = DataNode;
		using pointer           = DataNode*;
		using reference         = DataNode&;

		std::shared_ptr<DataNode> node_;
	 public:
		NameIterator() :
			node_{std::make_shared<DataNode>()} {};

		explicit NameIterator(DataNode &node) :
			node_{std::make_shared<DataNode>(node)}
		{};

		NameIterator &operator++();

		const NameIterator operator++(int);

		bool operator==(const NameIterator &it) const { return node_->curren_xml_node_ == it.node_->curren_xml_node_; };

		bool operator!=(const NameIterator &other) const { return !(*this == other); };

		reference operator*() const { return *node_; }

		pointer operator->() { return node_.get(); }

	};

	DataNode &operator=(const DataNode &d) = default;

	explicit operator bool() const;

	DataNode &operator++();

	const DataNode operator++(int);

	DataNode &operator--();

	const DataNode operator--(int);

	bool operator==(const DataNode &d) const;

	bool operator!=(const DataNode &other) const;

	reference operator*();

	pointer operator->();

	/*
	 *    .
	 */
	[[nodiscard]] const char *GetName() const;

	/*
	 *  ,    .
	 */
	[[nodiscard]] bool IsEmpty() const;

	/*
	 *  ,     .
	 */
	[[nodiscard]] bool IsNotEmpty() const { return !IsEmpty(); };

	/*
	 *     .   
	 *   ,    .
	 */
	[[nodiscard]] const char *GetValue(const std::string &key = "") const;

	/*
	 *     .
	 */
	void GoToRadix();

	/*
	 *     ,     .
	 */
	void GoToParent();

	/*
	 *        .
	 */
	bool HaveChild(const std::string &key);

	/*
	 *     key,    .
	 */
	bool GoToChild(const std::string &key);

	/*
	 *     key,    .
	 */
	bool GoToSibling(const std::string &key);

	/*
	 *    .
	 */
	bool HavePrevious();

	/*
	 *     .
	 */
	void GoToPrevious();

	/*
	 *    .
	 */
	bool HaveNext();

	/*
	 *     .
	 */
	void GoToNext();

	/*
	 *    .
	 */
	[[nodiscard]] iterators::Range<DataNode> Children();

	/*
	 *      key,
	 */
	[[nodiscard]] iterators::Range<NameIterator> Children(const std::string &key);

};

} //parcer_wrapper

#endif // PARSER_WRAPPER

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
